$DECLARE
$NOTRUNCATE
C
	PROGRAM DRIFVEL
C
C
C	This program uses DRIFT sources located by the program SKY to 
C	calculate ionospheric drift velocity vectors.
C
C	PROGRAMMER:  Terence W. Bullett
C	DATE:        27 April 1988
C	FORTRAN:     Microsoft FORTRAN V4.10
C	HARDWARE:    IBM AT compatable w/ 60Meg Hard Disk
C	             Pertec 9 track tape drive
C	             Okidata 192/192+ printer
C
C	MAJOR MODIFICATIONS:
C	    18 January 1989 - All source points over a given time are 
C	                      availible for calculation.
C	                    - All sources for a given time period and
C	                      over specified limits in height and frequency
C	                      are fit to a single velocity vector.
C	                    - Data can be availible for a large number of
C	                      heights and frequencies in the ionosphere.
C	                    - Unused statisitcal information is removed.
C	                    - Eliminatiion of the 4 frequency/height limit
C	                      by removal of that variable from the program.
C    
C	   22 February 1989 - Input/Output routines improved to include
C	                      a control file containing all necessary
C	                      values to run the program
C	                    - An additional output channel (AUX) is provided
C	                      for diskfile output.
C	                    - Additional printing formats provided, including
C	                      one which contains most all the velocity results.
C
C	   08 August 1989   - Remove the 'Noise Shadow' Doppler lines appearing
C	                      when the Drift phase code (X=4) is active.
C	                    - Add a 'Line-of-Sight' velocity which will use
C	                      sources only in a particular direction and 
C	                      within a given beamwidth.  This is implemented
C	                      using the /BOUNDS/ data.  Data can be partitioned
C	                      not only by Frequency and Height, but by Arrival
C	                      Angle also.
C	                    - Allow for overlap in the /BOUNDS/, ie one source
C	                      can be in more than one FRAB (up to NOLAP) at 
C	                      a time. 
C	                    - Add to the velocity output file the least squares
C	                      velocity which is the fit of ALL selected sources
C	                      to a particular velocity.  It is statistic #7 in
C	                      the /GRPVEL/ group velocity block.
C
C	   06 December 1989 - Converted to FORTRAN v. 5.0, clean up variables.
C	                    - Add Sondrestrom to station list.
C	                    - Re-organize file structure to include VELHDG for
C	                      the Density Graph routines.
C	                    - Clean up OUTPUT routines.
C
C	   28 December 1989 - Version 3.1 Control File -- Placed file names
C	                      AFILE,LFILE,IFILE on a separate line. Previous
C	                      line was too long.
C	                    - Improve consistency in the use of the NSELXED
C	                      parameter and the SELINDX array to allow both
C	                      INDIVEL3D and INDIVELOS to function properly.
C	                    - Add array dimension parameters NDLM (# Doppler
C	                      Lines for Mapdata), NIND (# of Individual
C	                      Velocities) and NPDF (# of Probability 
C	                      Distribution Function bins).
C
C
C ************************* SUBROUTINES *********************************
C
C...In file VELH.FOR    - Main Calculation Routines.
C	FILTER      -- Filters /MAPDATA/ on the basis of SNR,Doppler, etc.
C	               All source selections that can be made A PRIORI 
C	               (without having all sources available or having a
C	               velocity calculated) are performed here.
C	PARTITION   -- The grouping of the data according to the /BOUNDS/
C	               is performed by this Integer Function.
C	DOPRES      -- This Real Function uses DGS256 PREFACE data to determine
C	               the Doppler resolution of the current /MAPDATA/
C	SORTSOURCE  -- Sorts the /SOURCES/ after they are all availible and
C	               CHOOSEn but before calculations are to be performed.
C	SAVESOURCE  -- Moves selected sources from /MAPDATA/ to /SOURCES/
C	               while performing the necessary conversions.  The 
C	               values in the /SOURCES/ block are less dependent upon
C	               header variables and are more representative of the
C	               actual attributes of the variables measured. 
C	CHOOSE      -- Given all data for a specified time period, this 
C	               routine selects that data (using /BOUNDS/) which
C	               will be used in the calculation of INDIVIDUAL and
C	               GROUP velocities.
C	RECHOOSE    -- Given that INDIVEL3D has failed with the data as
C	               arranged by CHOOSE, this routine changes that choice,
C	               hopefully for the better.
C	ADROK       -- Logical function returning the operational status
C	               of the AutoDrift program used to take the current
C	               data. 
C	IND3DOK     -- Logical Function determines if the actions performed
C	               by INDIVEL3D are acceptable.  If not, sources are
C	               deleted.
C	TIMEBREAK   -- Logical function which determines when to split
C	               the input data stream in time.
C	CORRECT     -- Ionospheric effects (such as refraction) are
C	               compensated for here.  Also, compensation for
C	               system/sampling effects implemented here, such as
C	               elimination of Hanning spreading.
C	INDIVEL3D   -- 3 Dimensional Individual Velocity Calculation.
C
C	GROUPVEL    -- Group Velocity Calculation.  Statistical data 
C	               reduction performed at this stage, including means,
C	               medians,standard deviations and quartiles.
C	PDF         -- The Probability Distribution Functions (PDFs) for
C	               the /VELOCITY/ data are calculated here, along with 
C	               the Most Probable values.
C	INITIALIZE  -- Resets necessary variables between calculations.
C	VELCALC     -- Oversees the velocity calculations.
C	CHECKINDV   -- Checks the sequence of individual velocities AFTER
C	               they have all been computed.
C	ZEROSUMS    -- Initializes the INDIVEL(LOS/3D) intermediate results.
C	ADDTOSUMS   -- Accumulates sources into the /SUMS/ 
C	SOLVESYS    -- Calculates a 3D velocity from the /SUMS/.
C	TRUEHGT     -- Real Function converts the virtual height given in 
C	               the drift data to a true height of reflection.
C	               ** UNIMPLIMENTED **
C	MDOPLINE    -- Integer Function determines the Maximum Doppler Line
C	               from the current DGS256 preface.
C	RMSESQ      -- Real function calculates the RMS least-square error
C	               for the least square velocity calculated by SOLVESYS.
C	VOLCHECK    -- Logical function checks the volume between the first 
C	               three points INDIVEL3D is to use for the velocity 
C	               calculation.  Changes the order of points if there is
C	               insufficent volume.  Returns .FALSE. if re-arranging
C	               the points does not help.
C	WEIGHT      -- Determines the relative weight for a source, given the
C	               different weighting options.
C
C...In file VELH2.FOR  - Additional stuff.
C	REPORT      -- Indicates an error, its location, and allows for user
C	               termination of the program.
C	PARSETIME   -- Changes time values from INTEGER*2(11) to REAL*8.
C	CTIMELTT    -- Converts time from LTT to CTT formats.
C	PARCTIME    -- Convtets time from CHARACTER*17 to REAL*8 formats.
C	CARTESIAN   -- Converts from spherical to cylindrical coodrinates.
C	SETSTATION  -- Initializes the info in the /STATION/ block.
C	IROUND      -- Rounds a real number to the nearest integer.
C	ISGN        -- Function returns 1 or -1 depending in sign of argument.
C	GETINT      -- Function allows user to input a bounded integer.
C	INITOPTI    -- Initialize Oki192 printer for Optifont.
C	OPTI        -- Returns Optifont character for given integer.
C	OPTICOUNT   -- Returns an Optifont character == log10 of the argument.
C	SIND   \
C	COSD     \
C	TAND      > -- Trig functions that work with degrees, not radians.
C	ASIND    /
C	ACOSD  /
C	ATAND2
C
C...In file VELHIO.FOR  - Input/Output Routines.
C	INPUTDATA      -- Reads the operator's inputs with menu-driven options.
C	                  Sets variables in /CONTROL/, /SELECT/, and /STATION/.
C	                  Opens necessary files also.
C	WRITEGROUPDATA -- Writes the full contents of the /GRPVEL/ block 
C	**READGROUPDATA-- Reads the full contents of the /GRPVEL/ block
C	              *** This routine located elsewhere, not needed here.
C	READMAPDATA    -- Reads data from the input file into the /MAPDATA/ 
C	READCTRLFILE   -- Reads all necessary input info. from a file instead 
C                       of from the keyboard.
C	WRITECTRLFILE  -- Writes all necessary input info. to a file for 
C	                  later reading. 
C	OUTPUT         -- Directs all output and manages page formatting for
C	                  data going to the printer.
C	LISTDATA       -- Output to files and/or printer managed here
C	AUXDATA        -- Output to the AUX device managed here.
C	PRTHEAD        -- Outputs a header to devices as appropriate.
C	DUMPSOURCE     -- Prints all sources in the /SOURCES/ block.
C
C...In file VELHDG.FOR  - Density Graphing routines.
C	GRAPHEAD    -- Prints header for Density Graphs.
C	BLANKLINE   -- Clears a line for printing, and adds dots as needed.
C	SCALEDGR    -- Sets the /VELPDFS/ limits for the DENGRAPH routine.
C	DENGRAPH    -- Density Graph printing coordinated here.
C
C...In file VELHNU.FOR  - New Routines, under devlopment.
C	SETBOUNDS      -- Sets up the /BOUNDS/ common block.
C	INITIOFLAGS    -- Sets the /CONTROL/ flags to direct the output of each
C	                  bin to all devices.
C	INDIVELOS      -- Individual Velocity Line-Of-Sight routine.
C	INDLOSOK       -- Checks progress of INDIVELOS
C	VELLOS         -- Solves for the Vlos after INDIVELOS has run.
C	VLOSRMS        -- Least square error in the Vlos calculation.
C	SIGHTLINE      -- Determines dot product of current LOS and another
C	                  vector (source position).
C	STATGRAPH      -- Places statistics in Density Graph **UNIMPLEMENTED**
C	SUNRIZE        -- Sunrise/Sunset routine, **UNIMPLEMENTED**
C
C...In file MATH.FOR    - Numerical and Statisitcal Routines, taken mostly
C                           from "Numerical Recipies".
C===========================================================================	
C    Parameters -- typically used as array dimensions, therefore maxima.
C	NIND    - Number of INDividual velocities.
C	NFRAB   - Number of Frequency-Range-Angle Bins.
C	NSTAT   - Number of STATistics to calculate from individual velocities.
C	NSS     - Number of Saved Sources.
C	NPDF    - Number of Probability Distribution Function bins.
C	NDLM    - Number of Doppler Lines, Maximum for each spectra.
C	NOLAP   - Number of OverLAPping FRABs allowed.
C	
C===========================================================================	
C
C.....Common Blocks
C   /CONTROL/   Refer to the INPUTDATA routine in VELHIO.FOR for a more
C	          complete explination of these variables.
C	CHKADR       -  Check AutoDrift operation
C	FILE         -  Flag to select between file and printer.
C	IGTYPE       -  Select type of graph to be generated, None if = 0
C	IGCONT       -  Determine contents of desired graph.
C	IGOPTN       -  Determine options for selected type of graph.
C	IGCLEAN      -  Determine the level of cleaning for Density Graphs.
C	ILIST        -  Listing operation selector, none if = 0
C	IAUX         -  Auxilary listing operation selector, none if = 0
C	IGROUP       -  Grouping options, in time (see INPUTDATA,TIMEBREAK)
C	IWEIGH       -  Weighting options (see INPUTDATA, WEIGHT)
C	ISORT        -  Sorting options (see INPUTDATA, SORTSOURCE)
C	IFILTER      -  Filtering options (see INPUTDATA, FILTER)
C	ICORRECT     -  Correction options (see INPUTDATA, CORRECT)
C	ICHOOSE      -  Data Choice options (see INPUTDATA, CHOOSE)
C	IU           -  The Input Unit # for reading MAPDATA
C	PU           -  The outPut Unit number, can be file or printer
C	AU           -  Auxilary Unit number for output, must be file.
C	INFILE       -  Input file name
C	LFILE        -  List file name
C	AFILE        -  Auxilary file name
C	GFLAG(NFRAB) -  Logical flag map indicating which FRAB's to GRAPH.
C	LFLAG(NFRAB) -  Logical flag map indicating which FRAB's to LIST.
C	AFLAG(NFRAB) -  Logical flag map indicating which FRAB's to AUX.
C
C
C  /GRPVEL/        -  Statistical data for the Group Velocity in NFRAB
C	                possible Frequency-Range-Angle Bins.
C	                NFRAB and NSTAT are
C	                program parameters.
C	                The second index indicates which value is contained. 
C GVZ(NFRAB,NSTAT) -  Statistics for the Group Velocity Z-component.
C	     , 1)    -  Mean
C	     , 2)    -  Standard Deviation
C	     , 3)    -  Median
C	     , 4)    -  Upper quartile
C	     , 5)    -  Lower quartile
C	     , 6)    -  Most probable
C	     , 7)    -  Least Square fit to ALL data. 
C   8) -,NSTAT)    -  For future use  
C GVH(NFRAB,NSTAT) -  Statistics for the Group Velocity Horizontal magnitude.
C GAZ(NFRAB,NSTAT) -  Statistics for the Group Velocity horizontal AZimuth.
C GVX(NFRAB,NSTAT) -  Statistics for the Group Velocity X component.
C GVY(NFRAB,NSTAT) -  Statistics for the Group Velocity Y component.
C GSQ(NFRAB,NSTAT) -  Statistics for the velocity root mean square error.
C GVL(NFRAB,NSTAT) -  Statistics for the Group Velocity Line-of-Sight.
C GSL(NFRAB,NSTAT) -  Statistics for the LOS velocity root mean square error.
C NGVEL(NFRAB)     -  Number of measurements at each FRAB.
C NGVELOS(NFRAB)   -  Number of measurements at each FRAB.
C
C  /BOUNDS/      -  Data in this common block define the limits in
C                   Frequency, Range and Arrival Angle for each of the
C	              NFRAB's. 
C  NFRABS        -  Number of FRAB's defined.  Must be <= NFRAB.
C  TRUEHEIGHT    -  Logical flag selecting between true and virtual height
C	              selection of the data.  *** TRUE HEIGHT NOT YET AVAILABLE ***
C  BFRQ(NFRAB,2) -  Bounds upon the FReQuency, upper ,1) and lower ,2)
C  BRNG(NFRAB,2) -  Bounds upon the RaNGe, upper ,1) and lower ,2)
C  BANG(NFRAB,4) -  Bounds on the ANGles of arrival for this bin.
C                   ,1) = Zenith angle of the Line-of-Sight direction (deg).
C                   ,2) = Azimuth of the Line-of-Sight direction (deg).
C                   ,3) = Half 'Beamwidth' in Zenith (deg).
C                   ,4) = Half 'Beamwidth' in Azimuth (deg).
C  NBSRC(NFRAB)  - Number of sources in each FRAB
C
C
C  /MAPDATA/   Data read in from input file (SKY output)
C	NMAP      	-  Record number from original tape (useless here)
C	IFF         -  Frequency/height number (1-4)
C	IPREF(89)   -  Digisonde Preface characters
C	FREQ        -  Frequency of measurement MHz
C	RANGE       -  Virtual height of the data, Km
C	GAIN        -  Receiver attenuation, dB
C	MPAMP       -  Most Probable AMPlitude, dB
C	MAXAMP      -  Maximum Amplitude, dB
C	ZMAX        -  Maximum Zenith angle, degrees
C	NUMSRC      -  # of sources that follow
C	YMAP(NDLM)   -  Y coordinates of sources
C	XMAP(NDLM)   -  X coordinates of sources
C	MAPAMP(NDLM) -  Amplitudes of sources, dB
C	MAPDOP(NDLM) -  Doppler channel # of sources
C	MAPRMS(NDLM) -  Source location error
C
C  /SELECT/  Values which effect the selection of CASES, SOURCES, VELOCITIES
C	MINSRC      -  Minimum # of sources to use for calculation
C	MINDOPP     -  Minimum Doppler # to use
C	MAXDOPP     -  Maximum Doppler # to use
C	MPOSERR     -  Maximum positional error to allow (degrees).
C	MCASESNR    -  Subcase maximum Signal - Noise case selection threshold.
C	MINLOBE     -  Lower threshold for source selection, realtive to
C	               subcase maximum amplitude.
C	MSRCSNR     -  Minimum SNR for each source.
C	VZMAX       -  Maximum vertical velocity to allow (m/s).
C	VHMAX       -  Maximum horizontal velocity to allow (m/s).
C	VERRMAX     -  Maximum % velocity error to allow.
C	VOLMIN      -  Minimum volume sources must occupy in space.

C  /SINDEX/  Index arrays for active (selected) /SOURCES/
C	NSELXED       - Number of active entries in SELINDX index.
C	SELINDX(NIND) - Index of those sources selected for INDIVEL3D/LOS.
C	                Contains location of source in /SOURCES/
C
C  /STATION/  Station dependent values
C	ILOC        -  Location # (1-6)
C	LOCATION    -  Name of station (text)
C	SID         -  3 character station ID code, should match preface.
C	GEOLAT      -  Geographic Latitide, Degrees North
C	GEOLON      -  Geographic Longitude, Degrees East
C	BLAT        -  Magnetic Latitude
C	BLONG       -  Magtetic Longitude.
C	VELHMAX     -  Maximum horizontal velocity magnitude for plots.
C	VELZMAX     -  Maximum absolute vertical velocity for plots.
C	GYROFREQ    -  Gyrofrequency in MHz.
C	DIPANGLE    -  Dip angle of the station, Degrees.
C
C  /SOURCES/ Information on each source required for determining the
C	       velocity vector.  NSS is the parameter for the maximum
C            number of saved sources.
C	NSRCRS         - Current number of sources  (INTEGER*4)
C	NSOVER          - Current number of excess sources over NSS (INTEGER*4).
C	THETAS(NSS)    - INTEGER*2 arrival elevation angle*10, from Zenith.
C	PHIS(NSS)      - INTEGER*2 arrival azimuth*10, from magnetic North, 
C                      increasing to the EAST, like Navigational coordiantes.
C                      *** NOTE ***  This is opposite SKYMAP data conventions. 
C	ERMS(NSS)      - INTEGER*1 source RMS positional error.
C	AMPS(NSS)      - INTEGER*1 amplitude of the source,dB.
C	NOIS(NSS)      - INTEGER*1 noise (MPA) of the source,dB.
C	VRS(NSS)       - Radial velocity of the source. (REAL*4)
C	BINS(NSS,NFRAB)- INTEGER*1 The /BOUNDARY/ bins for this source.
C	                 Up to NFRB are possible
C	ACTIVE(NSS)    - LOGICAL*1 flag indicating selection of this source.
C 
C  /SUMS/   Necessary components for calculating individual velocities
C	      Values here are intermediate steps in the Individual Velocity
C	      vector Least-square error calculation  (See INDIVEL3D/LOS).
C	SUMW   - Sum of the weights.
C	SUMWVM - Sum of the product of the weights*velocities.
C	SUMWAV - Sum of weights*LOS directions*velocities (for Vlos).
C	SUMWAA - Sum of weights*LOS directions (for Vlos).
C	R(3,3) - The 9 component sums of the least-square fit. 
C	RVM(3) - The sum of the (R.Vm) term in the least-square fit.
C	NSUMS  - Number of data points involved in the above sums.
C
C  /VELOCITY/ The calculated individual velocity components.
C             NIND is a parameter for the maximum number of 
C	        individual velocities allowed.
C	NIVEL3D    -  Number of 3D individual velocities in the arrays.
C	NIVELOS    -  Number of LOS individual velocities in the arrays.
C	INDIVBIN   -  Freq-Range-Angle Bin number for these velocities.  
C	VX(NIND)   -  Velocity X components (m/s).
C	VY(NIND)   -  Velocity Y components (m/s).
C	VZ(NIND)   -  Velocity Z components (m/s).
C	ESR(NIND)  -  Root Mean Squared Error in the fit of the sources to
C	              the (VX,VY,VZ) uniform velocity.
C	DETR(NIND) -  The determinant of the R array in /SUMS/ for each
C	              individual velocities.  Used as an indication of
C	              the stability of the calculations.
C	VL(NIND)   -  The Line-of-Sight Velocity
C	ESRL(NIND) -  The Root Mean Squared error of the sources to the
C	              Line-of-Sight velocity.
C
C  /VELPDFS/ Similar to the information in the /VELOCITY/ block, the
C	       Individual Velocities are represented as Probability
C	       Distribution Functions in Cartesian and Cylindrical coordinates.
C	       See the subroutine MAKEPDF for details.
C	LIMV(1,3)    - Vx for the PDF: Low, High, # bins, #bins/tickmark.
C	    (2,3)    - Vy                   "
C	    (3,3)    - Vz                   "
C	    (4,3)    - Vhorizontal				"
C	    (5,3)    - Vazimuth					"
C	    (6,3)    - Vlos  					"
C	KPDF(1,NPDF) - The PDF for Vx.
C	    (2,NPDF) - The PDF for Vy.
C	    (3,NPDF) - The PDF for Vz.
C	    (4,NPDF) - The PDF for Vhorizontal.
C	    (5,NPDF) - The PDF for Vazimuth.
C	    (6,NPDF) - The PDF for Vlos
C	STEP(6)      - The value step for each of the 6 parameters above
C	
C  /TIME/
C	STARTTIME, ENDTIME, NOWTIME, LASTTIME
C	ATIME       - Data Accumulation time = time over which data is
C	              averaged/smoothed. (seconds)
C	ACCSTIME    - The accumulation (averaging) start time. 
C	CSTIME,CETIME,CNTIME,CCSTIME,CLTIME - Times in CHARACTER*17 format
C	                         YYYY DDD HH:MM:SS 
C	NR2DO       - The number of Subcases to process if ENDTIME unspecified.
C
C
$INCLUDE:'COMMON\PARAMS.CMN'
$INCLUDE:'COMMON\BOUNDS.CMN'
$INCLUDE:'COMMON\CONTROL.CMN'
$INCLUDE:'COMMON\GRPVEL.CMN'
$INCLUDE:'COMMON\MAPDATA.CMN'
$INCLUDE:'COMMON\SELECT.CMN'
$INCLUDE:'COMMON\SINDEX.CMN'
$INCLUDE:'COMMON\SOURCES.CMN'
$INCLUDE:'COMMON\STATION.CMN'
$INCLUDE:'COMMON\TIME.CMN'
$INCLUDE:'COMMON\VELOCITY.CMN'
$INCLUDE:'COMMON\VELPDFS.CMN'
C
C
C.....Program variables
	INTEGER NSKIP,NSUBCASE,NTOSS,MDU
	LOGICAL EOF50,CALCVEL,AUTODR
C.....Functions
	LOGICAL TIMEBREAK,ADROK
	REAL*8 PARSETIME
	CHARACTER*17 CTIMELTT
C
C
C.....Determine program control parameters.
	CALL INPUTDATA()
	CALL SCALEDGR(IGTYPE,IGCONT)
C 
C.....Start searching for the first record desired.
	NSKIP = 0
	MDU = IU
	IF (STARTTIME.EQ.0) THEN
	   WRITE(*,*) ' Starting at begining of file ...'
	   CALL READMAPDATA(MDU,EOF50)
	   LASTTIME = 0.0
	   NOWTIME = PARSETIME(IPREF)
	   LASTTIME = NOWTIME
	   CNTIME = CTIMELTT(IPREF)
	   CLTIME = CNTIME
	ELSE
 11	   CALL READMAPDATA(MDU,EOF50)
	   IF(EOF50) GO TO 1450
	   LASTTIME = NOWTIME
	   CLTIME = CNTIME
	   NOWTIME = PARSETIME(IPREF)
	   CNTIME = CTIMELTT(IPREF)
	   IF (NOWTIME.LT.STARTTIME) THEN
	      WRITE(*,'(1X,2A)') CNTIME,' Skipping ...'
	      NSKIP = NSKIP + 1
	      GOTO 11
	   ENDIF
	   WRITE(*,133) CNTIME,NSKIP
	ENDIF
C
 133	FORMAT(' Found desired start time ',A20,'.  Skipped ',I4,
     +         ' Subcases.')
C
C.....Desired record has been found, start processing.
C
C	The sequence of data processing is to fill up the /SOURCES/ common
C	block with data for the interval specified in the /CONTROL/ block.
C	After all sources have been found, velocities are calculated.
C
	NSUBCASE = NSKIP + 1
C.....Date and time update, zero sources and prepare for calculations.
	CALL INITIALIZE()
C
C.....Start the loop which accumulates sources for the  desired period
C.....of time.
C
 1	CONTINUE
C
C........Determine if it is time to calculate a GROUP velocity.
	   CALCVEL = TIMEBREAK()
	   IF (CALCVEL) THEN
C...........Do all the velocity calculations
	      CALL VELCALC()
C...........Initialize things for the next calculation.
	      CALL INITIALIZE()
	   ENDIF
C
C........Continue with source accumulation
C........Check the operation of AUTODRIFT.
	   IF (CHKADR)  AUTODR = ADROK()
	   NTOSS = 0
	   CALL FILTER(NTOSS)
	   CALL CORRECT(NTOSS)
	   CALL SAVESOURCE()
	   WRITE(*,134) CNTIME,NSUBCASE,NUMSRC-NTOSS,NUMSRC
 134	   FORMAT(1X,A17,', subcase # ',I5,3X,I3'/',I3,' good sources.')
C
	   CALL READMAPDATA(MDU,EOF50)
	   NSUBCASE = NSUBCASE + 1
	   LASTTIME = NOWTIME
	   CLTIME = CNTIME
	   NOWTIME = PARSETIME(IPREF)
	   CNTIME = CTIMELTT(IPREF)
	   IF (EOF50) GOTO 1450
	IF ((NOWTIME.LE.ENDTIME).AND.(NSUBCASE.LE.NR2DO)) GOTO 1
C
 1450	CONTINUE
C.....Finalize calculations on any data waiting to be processed.
	CALL VELCALC()
	CALL OUTPUT(-1)
	WRITE (*,*) ' Terminating after ',NSUBCASE-1, ' Subcases.'
	CLOSE (UNIT=IU)
C
	IF (EOF50) WRITE(*,*) ' End of data from input file : ',INFILE
	END 
C
C  
C  =======================================================================
C
	SUBROUTINE VELCALC()
C
C	This routine directs all the work of velocity calculation.  It is
C	called whenever source accumulation is finished, and at the end of the 
C	program to finish up what is in the arrays.
C	Note that either INDIVELOS or INDIVEL3D can disturb the selected and
C	sorted sources in /SINDEX/, so /SINDEX/ must be updated before each
C	velocity calculation.
C
C.....Parameters
$INCLUDE:'COMMON\PARAMS.CMN'
C.....Common Blocks
$INCLUDE: 'COMMON\BOUNDS.CMN'
$INCLUDE: 'COMMON\control.CMN'
$INCLUDE: 'COMMON\select.CMN'
$INCLUDE: 'COMMON\SINDEX.CMN'
$INCLUDE: 'COMMON\VELOCITY.CMN'
$INCLUDE: 'COMMON\time.CMN'

C
	INTEGER IBIN,NFAIL,ntry,lsort
	LOGICAL RESTART,ONELINE,retry
	character*20 dbfile
	real vhmag,vhmag2
C
	ONELINE = .FALSE.
C
c	1990 123 12:23:05
	write(dbfile,108) 'X',ccstime(6:8),ccstime(10:11),
     +        ccstime(13:14),'.DAT'
	write(*,*) '   ',dbfile
	open (file=dbfile,unit=44,form='formatted',mode='write')
108	format(10a)

	DO 20 IBIN = 1, NFRABS
C........Loop through all BINS with sources in them.
	   IF (NBSRC(IBIN).GT.0) THEN
C...........Set the number of velocities to zero.
	      NIVELOS = 0
	      NIVEL3D = 0
	      CALL CHOOSE(IBIN)
	      WRITE(*,102) IBIN,NSELXED
	      CALL SORTSOURCE(IBIN)
C...........Calculate the Line-of-Sight velocity.
	      NFAIL = 0
 32	      CONTINUE
	         CALL ZEROSUMS()
cdbg	         CALL INDIVELOS(RESTART)
	      IF (RESTART) GOTO 32
C...........Calculate the 3D velocity vector. Must CHOOSE sources again.
	      CALL CHOOSE(IBIN)
	retry = .true.
	ntry = 0
cd 33	      CONTINUE
c	Calculate the least square velocity, adjust the weights,
C	and then generate the individual velocities.
	   call sortsource(ibin)
	   CALL ZEROSUMS()
cx	   lsort = isort
cx	   isort = 8
	   call indivel3d(restart)
c	   vhmag = sqrt(vx(nivel3d)**2 + vy(nivel3d)**2)
c	   write(*,*) ' Vhmag = ',vhmag
c	   write(*,*) ' Enter new Vhmag'
c	   read(*,*) vhmag2
	   
c	   vx(nivel3d) = vx(nivel3d) * vhmag2 / vhmag
c	   vy(nivel3d) = vy(nivel3d) * vhmag2 / vhmag 
c	   vz(nivel3d) = vz(nivel3d)
	   do 40 ntry = 1, 1
	      CALL CHECKINDV(IBIN,Retry)
	      vhmag = sqrt(vx(nivel3d)**2 + vy(nivel3d)**2)
	      write(44,*) ' Vo (x,y,z,h) = ',vx(nivel3d),vy(nivel3d),
     +             vz(nivel3d),vhmag
	      write(*,*) ' Vo (x,y,z,h) = ',vx(nivel3d),vy(nivel3d),
     +             vz(nivel3d),vhmag
	      CALL SORTSOURCE(IBIN)
	      CALL ZEROSUMS()
	      CALL INDIVEL3D(RESTART)
 40	   continue
cx	isort = lsort
cd	         IF (VOLCHECK()) THEN
cd	         ELSE
cd	            WRITE(*,*) ' Insufficient source volume.'
cd	         ENDIF
cd	      IF (retry) GOTO 33
C
	      CALL GROUPVEL(IBIN)
C...........Output results here.
	      CALL OUTPUT(IBIN)
	      ONELINE = .TRUE.
	   ENDIF
 20	CONTINUE
	close (unit=44)
C
C.....Make sure OUTPUT has been called at least once this call.
	IF (.NOT.ONELINE) THEN
	   CALL OUTPUT(0)
	   WRITE(*,102) 0, 0
	ENDIF
 102	FORMAT(' Bin #',I3,' -- ',I5,' Sources selected.')
	RETURN
	END
C		
C  =======================================================================
C
	LOGICAL FUNCTION ADROK()
C
C	Checks the functioning of AUTODRIFT on the Digisonde 256 as best
C	as can be done from the data.
C
C	                    ***** UNIMPLIMENTED *****
	ADROK = .TRUE.
	RETURN
	END
C
C  =======================================================================
C
	SUBROUTINE CHECKINDV(IBIN,REtry)
C
C	Checks the results of the INDIVEL3D routine AFTER it has finished
C	running for the acceptability of the results.  Unacceptable results
C	require the re-running of INDIVEL3D from scratch.
C	RECHOOSE for RESTART of INDIVEL3D is managed here.
c
c	The deviation of each source from the estimated velocity is
C	calculated here, and the sources are re-weighted using this.
c	the /VELOCITY/ arrays have the NIVEL3D individual velocities.
c
c
C.....Parameters
$INCLUDE:'COMMON\PARAMS.CMN'
C.....Common Blocks
$INCLUDE:'COMMON\SOURCES.CMN'
$INCLUDE:'COMMON\CONTROL.CMN'
$INCLUDE:'COMMON\SINDEX.CMN'
$INCLUDE:'COMMON\velocity.CMN'
C
	INTEGER IBIN,i,isrc
	LOGICAL REtry
	REAL ermslast,theta,phi,x,y,z,vxsq,vysq,vzsq,vm,verr
c
	ibin = ibin
	ermslast = esr(nivel3d)
	write(*,*) ' CHECKINDV -- Esr = ',ermslast
c
c	Determine the new source weights

	retry = .true.
	vxsq = vx(nivel3d) 
	vysq = vy(nivel3d) 
	vzsq = vz(nivel3d)
 
	do 20 i = 1, nselxed
	   isrc = selindx(i)
C	   Determine the (X Y Z) components of the unit position vector
C	   of ISRC from its arrival angles THETA and PHI
	   THETA = THETAS(ISRC)/10.0
	   PHI   = PHIS(ISRC)/10.0
	   VM    = VRS(ISRC)
	   CALL CARTESIAN(1.00,THETA,PHI,   X,Y,Z)
c	Exclude (effectively) those points which have |V.R| < 10 m/s
	   verr = abs(x*vxsq + y*vysq + z*vzsq)
	   if ((verr.lt.10.0).or.(verr.gt.80.)) devwgt(isrc) = 1.0e30
cxx	   devwgt(isrc) = max(abs(verr), 1.0)
 20	continue
C
	RETURN
	END
C
C  =======================================================================
C
	SUBROUTINE RECHOOSE(IBIN,NFAIL)
C
C	If INDIVEL3D has failed, re-select from /SOURCES/ for INDIVEL3D.
C	Re-build an index of selected sources for
C	quick reference from the large /SORUCES/ array.
C	A re-ordering of source sequence, if necessary, is appropriate here.
C	The /CONTROL/ parameter IRECHOOSE determines the re-selection.
C	   1 -- The first MINSRC sources are removed from the list
C	   2 -- A source near the beginning of the list is swapped for one
C     	     near the end of the list.  NFAIL determines which ones.
C	NFAIL is the number of times INDIVEL3D has failed with this data set.
C
C.....Parameters
$INCLUDE: 'COMMON\PARAMS.CMN'
C.....Common Blocks
$INCLUDE: 'COMMON\CONTROL.CMN'
$INCLUDE: 'COMMON\SELECT.CMN'
$INCLUDE: 'COMMON\SINDEX.CMN'
$INCLUDE: 'COMMON\SOURCES.CMN'

	INTEGER IBIN,I1,I2,NFAIL,ISWAP
C
	IBIN = IBIN
	NFAIL = NFAIL + 1
	WRITE(*,'(A,I2)') ' RECHOOSE = ',NFAIL
	IF (IRECHOOSE.EQ.1) THEN
C........Remove the first MINSRC sources from the SELINDX list.
	   NSELXED = NSELXED - MINSRC
	   IF (NSELXED.LE.0) THEN
	      NSELXED = 0
	      RETURN
	   ENDIF
	   DO 20 I1 = 1, NSELXED
	      SELINDX(I1) = SELINDX(I1+MINSRC)
 20	   CONTINUE
C
	ELSE IF (IRECHOOSE.EQ.2) THEN
C........Swap sources.  The one near the beginning is #MOD(NFAIL,MINSRC)
C........and the one near the end is NSELXED-NFAIL.
	   I1 = MOD(NFAIL,MINSRC) + 1
	   I2 = NSELXED - NFAIL
	   ISWAP = SELINDX(I1)
	   SELINDX(I1) = SELINDX(I2)
	   SELINDX(I2) = ISWAP
	ENDIF
C
	RETURN
	END
C
C  =======================================================================
C
	SUBROUTINE SORTSOURCE(IBIN)
C
C	Sort the sources in /SINDEX/ according to the ISORT parameter in 
C	the /CONTROL/ block.
C
C.....Parameters
$INCLUDE: 'COMMON\PARAMS.CMN'
C.....Common Blocks
$INCLUDE: 'COMMON\SOURCES.CMN'
$INCLUDE: 'COMMON\CONTROL.CMN'
$INCLUDE: 'COMMON\SINDEX.CMN'
C
	INTEGER I,IBIN,KS,IDX(NSS)
	REAL SORT(NSS)
C
	IBIN = IBIN
C.....Don't bother if sorting is not desired.
	IF (ISORT.EQ.0) RETURN
C
	IF (ISORT.EQ.1) THEN
C........Decreasing amplitude.
	   DO 15 I = 1, NSELXED
	      SORT(I) = -FLOAT(AMPS(SELINDX(I)))
  15	   CONTINUE
	ELSE IF (ISORT.EQ.2) THEN
C........Increasing amplitude
	   DO 16 I = 1, NSELXED
	      SORT(I) = FLOAT(AMPS(SELINDX(I)))
 16	   CONTINUE
	ELSE IF (ISORT.EQ.3) THEN
C........Decreasing absolute Doppler velocity.
	   DO 17 I = 1, NSELXED
	      SORT(I) = -ABS(VRS(SELINDX(I)))
  17	   CONTINUE
	ELSE IF (ISORT.EQ.4) THEN 
C........Increasing absolute Doppler velocity.
	   DO 18 I = 1, NSELXED
	      SORT(I) = ABS(VRS(SELINDX(I)))
  18	   CONTINUE
	ELSE IF (ISORT.EQ.5) THEN
C........Increasing RMS error.
	   DO 19 I = 1, NSELXED
	      SORT(I) = FLOAT(ERMS(SELINDX(I)))
  19	   CONTINUE
	ELSE IF (ISORT.EQ.6) THEN
C........Decreasing Zenith angle
	   DO 20 I = 1, NSELXED
	      SORT(I) = -FLOAT(THETAS(SELINDX(I)))
  20	   CONTINUE
	ELSE IF (ISORT.EQ.7) THEN
C........Increasing Zenith angle
	   DO 21 I = 1, NSELXED
	      SORT(I) = FLOAT(THETAS(SELINDX(I)))
  21	   CONTINUE
	ELSE IF (ISORT.EQ.8) THEN
C........Increasing DEVWGT()
	   DO 22 I = 1, NSELXED
	      SORT(I) = DEVWGT(SELINDX(I))
  22	   CONTINUE
	ENDIF
C
	KS = NSELXED				
	CALL INDEXX(KS,SORT,IDX)
C
C.....The IDX() array now contains the sorting order for the sources based
C.....on the values in SORT().  In order to arrange the SELINDX() array in
C.....that order, it is first copied to the SORT() array, then copied back
C.....in the order specified by IDX().  REAL's mixed with INTEGER's for  a
C.....savings of space.
C
	DO 50 I = 1, NSELXED
	   SORT(I) = SELINDX(I)
 50	CONTINUE
	DO 55 I = 1, NSELXED
	   SELINDX(I) = SORT(IDX(I))
 55	CONTINUE
C
	RETURN
	END
C
C  =======================================================================
C
	SUBROUTINE CHOOSE(IBIN)
C
C	Select those sources from /SOURCES/ which are to be used the
C	INDIVEL3D/LOS routines and translated into a vector velocies.
C	While selecting sources, build an index of selected sources for
C	quick reference into the large /SOURCES/ array.
C	A re-ordering of source sequence, if necessary, is appropriate here.
C	The /CONTROL/ parameter ICHOOSE can influence the operation of this
C	routine if necessary.
C
C	Also, set the INDIVBIN value in /VELOCITY/ to IBIN.
C
C	**** CURRENTLY ONLY CHOOSES *ALL* SOURCES IN THE SPECIFIED BIN ****
C
C.....Parameters
$INCLUDE:'COMMON\PARAMS.CMN'
C.....Common Blocks
$INCLUDE:'COMMON\BOUNDS.CMN'
$INCLUDE:'COMMON\CONTROL.CMN'
$INCLUDE:'COMMON\SINDEX.CMN'
$INCLUDE:'COMMON\SOURCES.CMN'
$INCLUDE:'COMMON\VELOCITY.CMN'

	INTEGER IBIN,ISRC,I
	LOGICAL INTHISBIN
C
	NSELXED = 0
	INDIVBIN = IBIN
	DO 10 ISRC = 1, NSRCRS
C........Search through the /SOURCE/BINS( , ) to see if ISRC is in IBIN.
	   INTHISBIN = .FALSE.
	   ACTIVE(ISRC) = .FALSE.
	   DO 5 I = 1, NFRABS
  	      INTHISBIN = (BINS(ISRC,I).EQ.IBIN).OR.INTHISBIN
 5	   CONTINUE
C
	   IF (INTHISBIN) THEN
	      NSELXED = NSELXED + 1
	      SELINDX(NSELXED) = ISRC
	      ACTIVE(ISRC) = .TRUE.
	   ENDIF
 10	CONTINUE
	RETURN
	END
C
C  =======================================================================
C
	SUBROUTINE INITIALIZE()
C
C	Resets all necessary variables between calculations.
C
C.....Parameters
$INCLUDE:'COMMON\PARAMS.CMN'
C.....Common Blocks
$INCLUDE:'COMMON\BOUNDS.CMN'
$INCLUDE:'COMMON\CONTROL.CMN'
$INCLUDE:'COMMON\SINDEX.CMN'
$INCLUDE:'COMMON\SOURCES.CMN'
$INCLUDE:'COMMON\TIME.CMN'
C
	INTEGER I
	LOGICAL ROUND
	CHARACTER*17 CTIME
	REAL*8 PARCTIME
C.....No more sources.
	NSRCRS = 0
	NSOVER = 0
	NSELXED = 0
	DO 10 I = 1, NFRAB
	   NBSRC(I) = 0
 10	CONTINUE
C.....Set TIMEs properly
C
C.....Get the integration start time correct.
	ROUND = IAND(IGOPTN,8).NE.0
      IF (ROUND) THEN
C........Round the accumulation start time to the previous period.
	   CCSTIME = CNTIME
	   CCSTIME = CTIME(5)
         ACCSTIME = PARCTIME(CCSTIME)
	ELSE
C........Just keep going. 
	   ACCSTIME = NOWTIME
	   CCSTIME = CNTIME
      ENDIF
	RETURN
	END
C
C  =======================================================================
C
	SUBROUTINE CORRECT(NTOSS)
C
C	Correct sources in /MAPDATA/ for ionospheric/instrumental errors
C
$INCLUDE:'COMMON\PARAMS.CMN'
$INCLUDE:'COMMON\CONTROL.CMN'
$INCLUDE:'COMMON\MAPDATA.CMN'
C
	INTEGER I,MAXA,MAXD,GHOSTD,NSIG,NTOSS
	INTEGER*2 SPECTR(-128:128)
	LOGICAL ISO,NEIGHBOR
	INTEGER MDOPLINE,ISGN
C
	IF (NUMSRC.LT.1) RETURN
C
C.....Check if X<>0.  Phase codes tend to produce 'images' of the desired
C.....signal at a given spectral offset from the main signal.  
C
	IF ((IPREF(44).NE.0).AND.(ICORRECT.EQ.1)) THEN
CCX	   WRITE(*,*) ' X = ',IPREF(44)
C........Phase code is on, search and eliminate Doppler Ghosts.
C........Find the strongest (in amplitude) Doppler line.
	   MAXD = 0
	   MAXA = 0
	   DO 5 I = -128,128
 5	   SPECTR(I) = 0
C
CCX	WRITE(*,101) (MAPDOP(I),I=1,NUMSRC)
CCX	WRITE(*,101) (MAPRMS(I),I=1,NUMSRC)

	   DO 20 I = 1, NUMSRC
	      SPECTR(MAPDOP(I)) = I
	      IF ((MAPAMP(I).GT.MAXA).AND.(MAPRMS(I).GE.0.0)) THEN
	         MAXA = MAPAMP(I)
	         MAXD = MAPDOP(I)
	      ENDIF
 20	   CONTINUE
C
C........Look for a signal shifted from the spectral limit toward the
C........strongest Doppler line.
	   GHOSTD = -ISGN(MAXD)*MDOPLINE() + MAXD
	   NSIG = 0
	   ISO = .FALSE.
	   DO 25 I = GHOSTD-2, GHOSTD+2
	      IF (SPECTR(I).NE.0) NSIG = NSIG + 1
	      ISO = ISO .OR. (SPECTR(I).EQ.0)
 25	   CONTINUE
C
	   IF (NSIG.GT.0.AND.ISO) THEN
C...........If these NSIG Doppler lines are continuous and isolated then
C...........they are ghosts.
	      DO 30 I = GHOSTD-2, GHOSTD+2
	         NEIGHBOR = (SPECTR(I+1).NE.0).OR.(SPECTR(I-1).NE.0)
	         IF ((SPECTR(I).NE.0).AND.NEIGHBOR) THEN 
                  MAPRMS(SPECTR(I)) = - MAPRMS(SPECTR(I))
	            NTOSS = NTOSS + 1
	         ENDIF
 30	      CONTINUE
	   ENDIF
CCX	WRITE(*,101) (MAPDOP(I),I=1,NUMSRC)
CCX	WRITE(*,101) (MAPRMS(I),I=1,NUMSRC)
	   IF (NTOSS.GT.0) WRITE(*,102) NTOSS
 101	   FORMAT (1X,32I4)
 102	   FORMAT (I5,' Doppler Ghost Lines Detected and Eliminated.')
	ENDIF
	RETURN
	END
C
C  =======================================================================
C
	SUBROUTINE FILTER(NTOSS)
C
C	Apply the criteria in the /SELECT/ block to decide which /MAPDATA/
C	sources to keep and which ones to discard.
C	Deleted sources have a Negative MAPRMS.
C
$INCLUDE:'COMMON\PARAMS.CMN'
$INCLUDE:'COMMON\MAPDATA.CMN'
$INCLUDE:'COMMON\SELECT.CMN'
C
C
	INTEGER I,MD,NTOSS,MDMAX,MDOPLINE
	LOGICAL AOK
C
C	Check ratio of the subcase maximum amplitude to the subcase noise
C	to see if the case is to be kept.  If not, toss all sources.
C
	IF (((MAXAMP-MPAMP).LT.MCASESNR).OR.NUMSRC.EQ.0) THEN
	   NTOSS = NUMSRC
	   NUMSRC = 0
	   RETURN
	ENDIF
C.....Passed the subcase test, now check each source.
	MDMAX = MDOPLINE()
	IF (MAXDOPP.LT.0) THEN
	   MDMAX = MDMAX - ABS(MAXDOPP)
	ELSE
	   MDMAX = ABS(MAXDOPP)
	ENDIF
C
	DO 20 I = 1, NUMSRC
	   MD = ABS(MAPDOP(I))
	   AOK = (MD.GE.MINDOPP).AND.(MD.LE.MDMAX).AND.
     +         ((MAPAMP(I)-MPAMP).GE.MSRCSNR).AND.
     +         (MAPRMS(I).LE.MPOSERR).AND.
     +         ((MAXAMP-MAPAMP(I)).LE.MINLOBE)
	   IF (.NOT.AOK) THEN
	      NTOSS = NTOSS + 1
	      MAPRMS(I) = -MAPRMS(I)
	   ENDIF
 20	CONTINUE
	RETURN
	END
C
C
C  =======================================================================
C
	INTEGER FUNCTION PARTITION(FREQ,RANGE,THETA,PHI,BINLIST)
C
C	Assign a PARTITION number to the data in the source defined by
C	FREQ,RANGE,THETA,PHI.  Increment all /BOUNDS/NBSRC(IFRAB) in
C	which this source lies.  PARTITION then becomes the number of
C	Frequency-Range-Angle Bins in which the source lies.  The FRAB
C	number(s) in which the source lies are in the array BINLIST().
C	Use information in the /BOUNDS/ block to make this judgment.
C	Match is made in Frequency, Range and Angle.
C ===============
C =            ***** UNIMPLEMENTED *****
C =	If the TRUEHEIGHT flag is set then the true height of reflection
C =	is used for partitioning instead of the virtual height (RANGE).
C ===================
C
C.....Parameters
$INCLUDE:'COMMON\PARAMS.CMN'
C.....Common Blocks
$INCLUDE:'COMMON\BOUNDS.CMN'
C
C
	INTEGER IBIN,BINLIST(NOLAP),LAP
	REAL FREQ,RANGE,THETA,PHI,RNG,TRUEHGT,FRM
	LOGICAL THISFBIN,THISHBIN,THISTBIN,THISPBIN
C
	PARTITION = 0
C	Convert Range to Km, Frequency to MHz.
	IF (TRUEHEIGHT) THEN
	   RNG = TRUEHGT(FREQ,RANGE)
	ELSE
	   RNG = RANGE/1000.
	ENDIF
	FRM = FREQ/1.0E6
C
	DO 10 IBIN = 1, NFRABS
	   THISHBIN = (RNG.GE.BRNG(IBIN,1)).AND.(RNG.LE.BRNG(IBIN,2))
	   THISFBIN = (FRM.GE.BFRQ(IBIN,1)).AND.(FRM.LE.BFRQ(IBIN,2))
	   THISTBIN = (ABS(THETA-BANG(IBIN,1)).LE.BANG(IBIN,3))
	   THISPBIN = (ABS(PHI  -BANG(IBIN,2)).LE.BANG(IBIN,4))
	   IF (THISFBIN.AND.THISHBIN.AND.THISTBIN.AND.THISPBIN) THEN
	      PARTITION = PARTITION + 1
	      LAP = MIN(PARTITION, NOLAP)
	      NBSRC(IBIN) = NBSRC(IBIN) + 1
	      BINLIST(LAP) = IBIN
	   ENDIF
 10	CONTINUE
C
	RETURN
	END
C
C  =======================================================================
C
	REAL FUNCTION TRUEHGT(F,R)
C
C	Determines the true height from the /PROFILE/ information and
C	the frequency F and range R
C
C	                   ***** UNIMPLEMENTED *****
	REAL F,R
C
	TRUEHGT = F
	TRUEHGT = R
	RETURN
	END
C
C  =======================================================================
C
	SUBROUTINE SAVESOURCE()
C
C	This routine transfers the necessary data from the /MAPDATA/ block
C	to the /SOURCES/ block with appropriate translation.  Each source
C	is assigned to various BINs by the PARTITION routine.
C
C
C.....Parameters.
$INCLUDE:'COMMON\PARAMS.CMN'
C
C.....Common Blocks
$INCLUDE:'COMMON\MAPDATA.CMN'
$INCLUDE:'COMMON\SOURCES.CMN'
$INCLUDE:'COMMON\BOUNDS.CMN'
C
C
	INTEGER I,ISRC,NBIN,IROUND,PARTITION,BINLIST(NOLAP)
	REAL DFR,DOPRES,DOPFRQ,TS,PS,AS,ES,NS,C,VS
	REAL ASIND,SIND,ATAND2
C
	DATA C /2.997924574E+8/
C
	IF (NUMSRC.LT.1) RETURN
C
	DFR = DOPRES()
	DO 30 ISRC = 1, NUMSRC
C........Negative MAPRMS means do not save this source.
	   IF (MAPRMS(ISRC).GE.0) THEN
C...........Translate from /MAPDATA/ format to /SOURCES/ format.
C	      /MAPDATA/: X is North, Y is West, PHI increases to the West.
C	                 Right Handed Coordinate System
C	      /SOURCES/: X is North, Y is East, PHI increases to the East.
C	                 Left Handed Coordinate System
C
	      TS = ASIND(SQRT(XMAP(ISRC)*XMAP(ISRC) + 
     +            YMAP(ISRC)*YMAP(ISRC))/20.0 * SIND(ZMAX))
	      PS = ATAND2(-YMAP(ISRC),XMAP(ISRC))
	      AS = MAPAMP(ISRC) + GAIN
	      NS = MPAMP + GAIN
	      ES = MAPRMS(ISRC)
	      IF (MAPDOP(ISRC).GT.0) THEN
	         DOPFRQ = DFR*(FLOAT(MAPDOP(ISRC)) - 0.5)
	      ELSE
	         DOPFRQ = DFR*(FLOAT(MAPDOP(ISRC)) + 0.5)
	      ENDIF
	      IF (FREQ.NE.0.0) THEN
	         VS = -C*DOPFRQ/(2.0*FREQ)
	      ELSE
	         VS = 0.0
	      ENDIF
C
	      NBIN = PARTITION(FREQ,RANGE,TS,PS,BINLIST)
	      IF (NBIN.GT.NOLAP) THEN
	         WRITE(*,102) NBIN,NOLAP
	         NBIN = NOLAP
	      ENDIF
C
	      IF (NBIN.GT.0) THEN
C..............Add this source to /SOURCES/, flag as belonging in NBIN bins.
	         NSRCRS = NSRCRS + 1
C..............Array boundary checking.
	         IF (NSRCRS.GT.NSS) THEN
	            NSRCRS = NSS
	            NSOVER = NSOVER + 1
	         ENDIF
C..............Save the source.
	         THETAS(NSRCRS) = IROUND(TS*10.0) 
	         PHIS(NSRCRS)   = IROUND(PS*10.0)
	         AMPS(NSRCRS)   = AS
	         NOIS(NSRCRS)   = NS
	         ERMS(NSRCRS)   = ES
	         VRS(NSRCRS)    = VS
	         ACTIVE(NSRCRS) = .FALSE.
	         devwgt(nsrcrs) = 1.0
	         DO 25 I = 1, NBIN
	            BINS(NSRCRS,I) = BINLIST(I)
 25	         CONTINUE
C              Zero the rest of the BINS.
	         DO 27 I = NBIN+1, NOLAP
	            BINS(NSRCRS,I) = 0
 27	         CONTINUE
	      ENDIF
	   ENDIF
 30	CONTINUE
C
	IF (NSOVER.GT.0) WRITE(*,101) NSOVER,NSS
 101	FORMAT(' SAVESOURCE --',I4,' Excess sources.  Limit =',I5)
 102	FORMAT(' SAVESOURCE --',I2,' FRAB Overlaps.  Limit =',I2)
	RETURN
	END
C
C  =======================================================================
C
	REAL FUNCTION DOPRES()
C
C	Determines from the preface IPREF and the sounding frequency FREQ
C	the Doppler Frequency Resolution, which can be used to translate the
C	Doppler numbers in /MAPDATA/ to frequencies.
C	Ex:  Given a Doppler # I and the correct value of DOPRES then:
C			Doppler Shift (Hz) = DOPRES*(I - 0.5)  I > 0
C			Doppler Shift (Hz) = DOPRES*(I + 0.5)  I < 0
C	                *** for a Digisonde 256.  
C
C.....For /MAPDATA/
$INCLUDE:'COMMON\PARAMS.CMN'
$INCLUDE:'COMMON\MAPDATA.CMN'
C
	INTEGER*1 NFREQS(16),NANTS(16),NHITES(16),IL,IT3,IN
	INTEGER IREP,NSMPLS,NPULSES
	REAL TINTEG
C
C.......Value =   0 1 2 3 4 5 6 7 8 9 A B C D E F
	DATA NFREQS/1,2,1,2,1,2,1,2,2,4,2,4,2,1,2,4/ 
	DATA  NANTS/1,4,4,8,4,8,1,4,1,4,4,8,4,8,1,4/
	DATA NHITES/2,1,2,1,2,2,8,1,2,1,2,1,2,2,8,1/
C 
	IL = IPREF(45)
	IT3 = IAND(IPREF(47),8)
	IN = IPREF(48)
	IREP = 50*2**((IPREF(49)+1)/2) 
	NSMPLS = 32*(2**IAND(IN,3))
	NPULSES = NANTS(IL+1)*NFREQS(IL+1)*NHITES(IL+1)
	TINTEG = FLOAT(NSMPLS*NPULSES)/FLOAT(IREP)
	DOPRES = (1+IT3)/TINTEG
	RETURN
	END
C
C  =======================================================================
C
	INTEGER FUNCTION MDOPLINE()
C
C	Determines from the preface IPREF the Maximum Absolute Doppler
C	Frequency Number, which can be used to determine if the
C	Doppler numbers in /MAPDATA/ are near the end of the spectrum.
C
C	                    Common Blocks
C.....For /MAPDATA/
$INCLUDE:'COMMON\PARAMS.CMN'
$INCLUDE:'COMMON\MAPDATA.CMN'
C
	INTEGER IN
C
	IN = IPREF(48)
	MDOPLINE = 16*(2**IAND(IN,3))
	RETURN
	END
C
C  =======================================================================
C
	LOGICAL FUNCTION TIMEBREAK()
C
C	Function to determine the time at which to stop accumulating
C	sources and calculate group velocities.
C
C.....Parameters
$INCLUDE:'COMMON\PARAMS.CMN'
C	                    Common Blocks
$INCLUDE:'COMMON\CONTROL.CMN'
$INCLUDE:'COMMON\TIME.CMN'
C
C
	TIMEBREAK = .FALSE.
	IF (IGROUP.EQ.1) THEN
	   TIMEBREAK = .TRUE.
	ELSE IF ((IGROUP.EQ.2).AND.(NOWTIME.NE.LASTTIME)) THEN
C........Case velocities
	   TIMEBREAK = .TRUE.
	ELSE IF ((IGROUP.EQ.3).AND.((NOWTIME-LASTTIME).GT.59)) THEN
C........Group velocities
	   TIMEBREAK = .TRUE.
	ELSE IF ((IGROUP.EQ.4).AND.((NOWTIME-ACCSTIME).GT.ATIME)) THEN
C........Specified interval velocities
	   TIMEBREAK = .TRUE.
	ENDIF
	RETURN
	END
C
C
C  =======================================================================
C
	SUBROUTINE GROUPVEL(IBIN)
C
C     From the individual velocity information in the /VELOCITY/ common
C     block, calculate a Group Velocity via statistics.  Calculations are
C	performed in both Cartesian and cylindrical coordinates for comparison.
C	The Line-of-Sight Group Velocity is also calculated.
C	PDF information for /VELPDFS/ is also generated at this point.
C
C	IBIN        - The BIN index as defined by the /BOUNDS/
C	DATA(NIND)  - Temporary array for performing statistics on the
C	              Individual velocities.
C
C	                      Common Blocks
C
C.....Parameters.
$INCLUDE:'COMMON\PARAMS.CMN'
C	
C.....Variables for the COMMON blocks
$INCLUDE:'COMMON\CONTROL.CMN'
$INCLUDE:'COMMON\GRPVEL.CMN'
$INCLUDE:'COMMON\SELECT.CMN'
$INCLUDE:'COMMON\TIME.CMN'
$INCLUDE:'COMMON\VELOCITY.CMN'
$INCLUDE:'COMMON\VELPDFS.CMN'
C
C
	INTEGER I,IBIN
	REAL DATA(NIND),DUM1,DUM2,DUM3,DUM4
	REAL ATAND2
C
C.....Zero the group velocity statistical information.
	DO 10 I = 1, NSTAT
	   GVZ(IBIN,I) = 0.0
	   GVH(IBIN,I) = 0.0
	   GAZ(IBIN,I) = 0.0
	   GVX(IBIN,I) = 0.0
	   GVY(IBIN,I) = 0.0
	   GSQ(IBIN,I) = 0.0
	   GVL(IBIN,I) = 0.0
 10	CONTINUE
	NGVEL(IBIN)   = NIVEL3D
	NGVELOS(IBIN) = NIVELOS
C
	IF (NGVEL(IBIN).GT.0) THEN
C........There is now at least 1 3D velocity to average.
C
C========Vertical Velocity Component.
	   DO 20 I = 1, NGVEL(IBIN)
	      DATA(I) = VZ(I)
 20	   CONTINUE
C........Mean, Standard Deviation.
	   CALL MOMENT(DATA,NGVEL(IBIN),GVZ(IBIN,1),DUM1,GVZ(IBIN,2),
     +            DUM2,DUM3,DUM4)
C........Median, Upper Quartile, Lower Quartile.
	   CALL MEDIAN4(DATA,NGVEL(IBIN),GVZ(IBIN,3),GVZ(IBIN,4),
     +             GVZ(IBIN,5))
C........Most Probable.
	   CALL PDF(DATA,NIVEL3D,3,GVZ(IBIN,6))
C........Least Squares.
	   GVZ(IBIN,7) = VZ(NIVEL3D)
C
C========Horizontal Velocity Magnitude
	   DO 21 I = 1, NGVEL(IBIN)
	      DATA(I) = SQRT(VX(I)*VX(I) + VY(I)*VY(I))
 21	   CONTINUE
C........Mean, Standard Deviation.
	   CALL MOMENT(DATA,NGVEL(IBIN),GVH(IBIN,1),DUM1,GVH(IBIN,2),
     +            DUM2,DUM3,DUM4)
C........Median, Upper Quartile, Lower Quartile.
	   CALL MEDIAN4(DATA,NGVEL(IBIN),GVH(IBIN,3),GVH(IBIN,4),
     +             GVH(IBIN,5))
C........Most Probable.
	   CALL PDF(DATA,NIVEL3D,4,GVH(IBIN,6))
C........Least Squares.
	   GVH(IBIN,7) = SQRT(VX(NIVEL3D)*VX(NIVEL3D) +
     +                           VY(NIVEL3D)*VY(NIVEL3D))
C
C========Horizontal Velocity Azimuth.
	   DO 22 I = 1, NGVEL(IBIN)
	      DATA(I) = ATAND2(VY(I),VX(I))
 22	   CONTINUE
C........Mean, Standard Deviation.
	   CALL MOMENT(DATA,NGVEL(IBIN),GAZ(IBIN,1),DUM1,GAZ(IBIN,2),
     +            DUM2,DUM3,DUM4)
C........Median, Upper Quartile, Lower Quartile.
	   CALL MEDIAN4(DATA,NGVEL(IBIN),GAZ(IBIN,3),GAZ(IBIN,4),
     +             GAZ(IBIN,5))
C........Most Probable.
	   CALL PDF(DATA,NIVEL3D,5,GAZ(IBIN,6))
C........Least Squares.
	   GAZ(IBIN,7) = ATAND2(VY(NIVEL3D),VX(NIVEL3D))
C
C========Horizontal Velocity X Component.
	   DO 23 I = 1, NGVEL(IBIN)
	      DATA(I) = VX(I)
 23	   CONTINUE
C........Mean, Standard Deviation.
	   CALL MOMENT(DATA,NGVEL(IBIN),GVX(IBIN,1),DUM1,GVX(IBIN,2),
     +            DUM2,DUM3,DUM4)
C........Median, Upper Quartile, Lower Quartile.
	   CALL MEDIAN4(DATA,NGVEL(IBIN),GVX(IBIN,3),GVX(IBIN,4),
     +             GVX(IBIN,5))
C........Most Probable.
	   CALL PDF(DATA,NIVEL3D,1,GVX(IBIN,6))
C........Least Squares.
	   GVX(IBIN,7) = VX(NIVEL3D)
C
C========Horizontal Velocity Y Component.
	   DO 24 I = 1, NGVEL(IBIN)
	      DATA(I) = VY(I)
 24	   CONTINUE
C........Mean, Standard Deviation.
	   CALL MOMENT(DATA,NGVEL(IBIN),GVY(IBIN,1),DUM1,GVY(IBIN,2),
     +            DUM2,DUM3,DUM4)
C........Median, Upper Quartile, Lower Quartile.
	   CALL MEDIAN4(DATA,NGVEL(IBIN),GVY(IBIN,3),GVY(IBIN,4),
     +             GVY(IBIN,5))
C........Most Probable.
	   CALL PDF(DATA,NIVEL3D,2,GVY(IBIN,6))
C........Least Squares.
	   GVY(IBIN,7) = VY(NIVEL3D)
C
C========Mean Square error of the calculated velocity to the data.
	   DO 26 I = 1, NGVEL(IBIN)
	      DATA(I) = ESR(I)
 26	   CONTINUE
C........Mean, Standard Deviation.
	   CALL MOMENT(DATA,NGVEL(IBIN),GSQ(IBIN,1),DUM1,GSQ(IBIN,2),
     +            DUM2,DUM3,DUM4)
C........Median, Upper Quartile, Lower Quartile.
	   CALL MEDIAN4(DATA,NGVEL(IBIN),GSQ(IBIN,3),GSQ(IBIN,4),
     +             GSQ(IBIN,5))
C........Most Probable -- No PDF for the error.
	   GSQ(IBIN,6) = 0.0
C........Least Squares.
	   GSQ(IBIN,7) = ESR(NIVEL3D)
	ENDIF
C
C
C
	IF (NGVELOS(IBIN).GT.0) THEN
C========Line-of-Sight Velocity Component.
	   DO 25 I = 1, NGVELOS(IBIN)
	      DATA(I) = VL(I)
 25	   CONTINUE
C........Mean, Standard Deviation.
	   CALL MOMENT(DATA,NGVELOS(IBIN),GVL(IBIN,1),DUM1,GVL(IBIN,2),
     +            DUM2,DUM3,DUM4)
C........Median, Upper Quartile, Lower Quartile.
	   CALL MEDIAN4(DATA,NGVELOS(IBIN),GVL(IBIN,3),GVL(IBIN,4),
     +             GVL(IBIN,5))
C........Most Probable.
	   CALL PDF(DATA,NIVELOS,6,GVL(IBIN,6))
C........Least Squares.
	   GVL(IBIN,7) = VL(NIVELOS)
C
C=====Line-of-Sight Error.
	   DO 27 I = 1, NGVELOS(IBIN)
	      DATA(I) = ESRL(I)
 27	   CONTINUE
C........Mean, Standard Deviation.
	   CALL MOMENT(DATA,NGVELOS(IBIN),GSL(IBIN,1),DUM1,GSL(IBIN,2),
     +            DUM2,DUM3,DUM4)
C........Median, Upper Quartile, Lower Quartile.
	   CALL MEDIAN4(DATA,NGVELOS(IBIN),GSL(IBIN,3),GSL(IBIN,4),
     +             GSL(IBIN,5))
C........Most Probable -- No PDF for the error.
	   GSL(IBIN,6) = 0.0
C........Least Squares.
	   GSL(IBIN,7) = ESRL(NIVELOS)
	ENDIF
C
	RETURN
	END
C
C   ====================================================================
C
	SUBROUTINE PDF(DARRAY,NPTS,IW,VMP)
C
C	This routine takes real data in DARRAY and produces a Probability
C	Distribution Function (PDF) which is placed into the arrays in the
C	 /VELPDFS/ Common Block.   The bin number containing the
C	most values, or the Most Probable Value VMP is calculated.
C	 In the case of equal VMP's, the HIGHEST value is reported.
C
C	Values outside the range (START,STOP) are placed in the first or
C	last bin of the PDF.
C
C	DARRAY  - REAL Input data points.
C	NPTS    - # of points in DARRAY.
C	IW      - Which PDF (1-6) to calculate.
C	VMP     - Most Probable Value.
C
C..........Parammeters 
$INCLUDE:'COMMON\PARAMS.CMN'
C..........Common Blocks
$INCLUDE:'COMMON\VELPDFS.CMN'
C
	INTEGER I,J,IMPV,NPTS,IW,NDF,IROUND
	REAL DARRAY(NPTS),VMP,START
C
	IF (IW.GT.6.OR.IW.LT.1) RETURN
C
	START = REAL(LIMV(IW,1))
	NDF   = LIMV(IW,3)
	IF (NDF.LE.0) RETURN
C
	DO 5 I = 1, NDF
	   KPDF(IW,I) = 0
 5	CONTINUE
C
C	Calculate the PDF.
	DO 10 I = 1, NPTS
	   J = IROUND((DARRAY(I) - START)/STEP(IW))
	   J = MIN(MAX(J,1),NDF) 
	   KPDF(IW,J) = KPDF(IW,J) + 1
 10	CONTINUE
C
C.....Find MPV.
	J = 0
	IMPV = 0
	DO 15 I = 1, NDF
	   IF (KPDF(IW,I).GE.J) THEN
	      J = KPDF(IW,I)
	      IMPV = I
	   ENDIF
 15	CONTINUE
	VMP = START + IMPV*STEP(IW)
	RETURN
	END
C
C  =======================================================================
C
	LOGICAL FUNCTION VOLCHECK()
C
C	This routine makes certain that the volume defined by the first
C	three points and the origin is at least /SELECT/VOLMIN.  If not, 
C	the order of the sources is altered to make this so.  If this 
C	condition cannot be met, no velocity is calculated, so this
C	function returns .FALSE. and sets NIVEL3D to zero
C
C.....Parameters
$INCLUDE:'COMMON\PARAMS.CMN'
C.....Common Blocks
$INCLUDE:'COMMON\SELECT.CMN'
$INCLUDE:'COMMON\SOURCES.CMN'
$INCLUDE:'COMMON\SINDEX.CMN'
C
	INTEGER I,J,I1,IS,ISWAP,INDX(3)
	REAL VOL,THETA,PHI,X,Y,Z,RRS(3,3)
	LOGICAL SINGULAR
C
	VOL = 0.0
	VOLCHECK = .FALSE.
	IF (NSELXED.LT.3) THEN
	   RETURN
	ENDIF
C
	DO 10 I = 1, 3
	   IS = SELINDX(I)
	   THETA = THETAS(IS)/10.0
	   PHI   = PHIS(IS)/10.0
	   CALL CARTESIAN(1.0,THETA,PHI,   X,Y,Z)
	   RRS(I,1) = X
	   RRS(I,2) = Y
	   RRS(I,3) = Z
 10	CONTINUE
C
	I = 0
C.....DO WHILE (VOL.LT.VOLMIN)
 20	CONTINUE
	   CALL LUDECOMP(RRS,3,3,INDX,VOL,SINGULAR)
C........Calculate the Determinant of the RRS matrix.
	   DO 25 J = 1, 3
	      VOL = VOL * RRS(J,J)
 25	   CONTINUE
	   VOL = 6.0*VOL
C
C........Return .TRUE. if all is well.
	   IF ((ABS(VOL).GE.VOLMIN).AND..NOT.SINGULAR) THEN
	      VOLCHECK = .TRUE.
	      RETURN
	   ENDIF
C........Insufficient volume between the first 3 points.
C........Swap a later source into one of the first 3 places.
	   I = I + 1
	   I1 = MOD(I,3) + 1
	   ISWAP = SELINDX(I1)
	   SELINDX(I1) = SELINDX(3+I)
	   SELINDX(3+I) = ISWAP
	   IS = SELINDX(I1)
	   THETA = THETAS(IS)/10.0
	   PHI   = PHIS(IS)/10.0
	   CALL CARTESIAN(1.0,THETA,PHI,   X,Y,Z)
	   RRS(I1,1) = X
	   RRS(I1,2) = Y
	   RRS(I1,3) = Z
	IF (I.LE.NSELXED) GOTO 20
C.....Can't meet the volume condition.
	VOLCHECK = .FALSE.
	RETURN
	END
C
C   ====================================================================
C 
	SUBROUTINE INDIVEL3D(RESTART) 
C 
C
C	3D INDIVIDUAL VELOCITIES -- LEAST-SQUARE-ERROR CALCULATION: 
C 
C                                            2
C    ESQ= SUM{ W(I)*[(V.R(I))-(-C*DF(I)/(2*F))]}
C          I
C         ------------------------------------------
C                   SUM W(I)
C                    I
C 
C  ESQ             = Least Square Error 
C  W(I)            = Weighting Factor
C  (V.R(I))        = Dot product of velocity vector V (Vx,Vy,Vz) and the
C	               unit position vector for each source R(I)
C  C               = Speed of light
C  DF(I)           = Measured Doppler shift 
C  F               = Sounding Frequency
C  -C*DF(I)/(2*F)) = The radial component of the measured velocity, in the
C                    direction of R(I).  This is the Doppler value in the
C	               /SOURCES/ block.  Refered to as Vm in the code.
C
C	The deriviatives of ESQ (with respect to Vx,Vy and Vz) are each
C	set to zero, yeilding three equations which can then be solved for
C	the three unknowns Vx,Vy and Vz.  The Least-Square error ESQ can
C	then be calculated.  NOTE: X is North, Y is East, Z is Up.  This is
C	a LEFT HANDED COORDINATE SYSTEM!!
C
C	The matrix equation below is the one to be solved for 
C	Vx,Vy and Vz.  X,Y and Z are the Cartesian components of the
C	unit (|R|=1) source position vector R = (X,Y,Z) and Vm is the measured
C	Line-of-Sight Doppler velocity.  W is the weight factor for each
C	data point in the calculation.
C	
C     |---                                 ---| |-  -|     |---       ---|
C     | sum(W*X*X)   sum(W*X*Y)   sum(W*X*Z)  | | Vx |     | sum(W*X*Vm) |
C     |                                       | |    |     |             |
C     | sum(W*X*Y)   sum(W*Y*Y)   sum(W*Y*Z)  | | Vy |  =  | sum(W*Y*Vm) |
C     |                                       | |    |     |             |
C     | sum(W*X*Z)   sum(W*Y*Z)   sum(W*Z*Z)  | | Vz |     | sum(W*Z*Vm) |
C     |---                                 ---| |-  -|     |---       ---|
C
C	    --OR--, in martix form, [R][V] = [RVm]
C
C	The calling of this routine is done after all desired sources have
C	been read into the array /SOURCES/.  Only those NSELXED sources in the
C	SELINDX() array are used in this calculation.  These are the sources
C	in the IBIN block defined by /BOUNDS/ and selected by the CHOOSE()
C	routine.
C
C	Sources are assumed to have been selected and sorted in the order
C	in which it is desired to insert them into the least-square error
C	calculation.  /SOURCES/ are not altered by this routine.
C
C	Singularity of the inverted [R] matrix is checked for, and the
C	routine IND3DOK checks to see if a given velocity has exceeded
C	the limits specified for Individual Velocities in /SELECT/.
C	If a velocity exceeds the /SELECT/ limits, it is deleted.
C
C..........Parammeters 
$INCLUDE:'COMMON\PARAMS.CMN'
C.....Common Blocks
$INCLUDE:'COMMON\CONTROL.CMN'
$INCLUDE:'COMMON\SELECT.CMN'
$INCLUDE:'COMMON\SINDEX.CMN'
$INCLUDE:'COMMON\SOURCES.CMN'
$INCLUDE:'COMMON\SUMS.CMN'
$INCLUDE:'COMMON\VELOCITY.CMN'
$INCLUDE:'COMMON\time.cmn'
C
C
C	Program Variables.
	INTEGER IS,ISRC
	REAL W,VV(3),X,Y,Z,VM,THETA,PHI,RMSERR
	REAL RMSESQ,WEIGHT
	REAL*8 DET
	LOGICAL SINGULAR,RESTART,OK,IND3DOK
	real vhhh,vdr
C
C
	RESTART = .FALSE.
	IF (NSELXED.LT.MINSRC) THEN 
	   NIVEL3D = 0
	   WRITE(*,*) ' INDIVEL3D -- Not enough sources.'
	   RETURN
	ENDIF
C 
C.....Add the first MINSRC - 1 sources to the /SUMS/
	DO 30 IS = 1, MINSRC - 1
	   ISRC = SELINDX(IS)
C........Calculate the Weight W for the current source ISRC.
	   W = WEIGHT(ISRC,IWEIGH)/devwgt(isrc)
C........Determine the (X Y Z) components of the unit position vector
C........of ISRC from its arrival angles THETA and PHI
	   THETA = THETAS(ISRC)/10.0
	   PHI   = PHIS(ISRC)/10.0
	   VM    = VRS(ISRC)
	   CALL CARTESIAN(1.00,THETA,PHI,   X,Y,Z)
	   CALL ADDTOSUMS(VM,W,X,Y,Z)
 30	CONTINUE
C
C.....Now MINSRC-1 sources have been placed into the /SUMS/.  Add one more
C.....and generate an Individual Velocity for these sources.
C.....For each additional source added to /SUMS/, another Individual Velocity
C.....is generated.   NIVEL3D is the # of velocities.  IS is the source index
C.....into /SINDEX/SELINDX. ISRC is the index into /SOURCES/.
C
	write(44,108) '"',ccstime,'"'
	write(44,110) '#','Vh','Vz','Erv','Det','Theta','Phi',
     +               'Vlos','Ers','Amp','Devwgt','V.R'
c
	IS = MINSRC - 1
	NIVEL3D = 0
	NIV3OVR = 0
C.....DO WHILE (IS.LE.NSELXED)
 35	CONTINUE
C........Add the next source into the /SUMS/
	   IS = IS + 1
	   ISRC = SELINDX(IS)
	   W = WEIGHT(ISRC,IWEIGH)/devwgt(isrc)
	   THETA = THETAS(ISRC)/10.0
	   PHI   = PHIS(ISRC)/10.0
	   VM    = VRS(ISRC)
	   CALL CARTESIAN(1.0,THETA,PHI,   X,Y,Z)
	   CALL ADDTOSUMS(VM,W,X,Y,Z)
C........Calculate a 3D velocity vector
	   CALL SOLVESYS(VV,DET,SINGULAR)
	   RMSERR = RMSESQ(VV)
C
	   OK = IND3DOK(VV,RMSERR,DET,RESTART,IS)
	   RESTART = RESTART.OR.SINGULAR
	   IF (RESTART) THEN
	      write(*,*) ' INDIVEL3D -- Restart Requested.'
	      RETURN
	   ENDIF
C
	   IF (OK) THEN
C...........Save this velocity in the /VELOCITY/ block
	      NIVEL3D = NIVEL3D + 1
	      IF (NIVEL3D.GT.NIND) THEN
	         NIVEL3D = NIND
	         NIV3OVR = NIV3OVR + 1
	      ENDIF
	      VX(NIVEL3D) = VV(1)
	      VY(NIVEL3D) = VV(2)
	      VZ(NIVEL3D) = VV(3)
	      ESR(NIVEL3D) = RMSERR
	      DETR(NIVEL3D) = DET
	   ELSE
C...........This source produced results which were not OK, so
C...........Remove the last source from the /SUMS/.  The current
C...........source IS will be skipped.
	      CALL ADDTOSUMS(VM,-W,X,Y,Z)
	   ENDIF

	vhhh = sqrt(vx(nivel3d)*vx(nivel3d)+vy(nivel3d)*vy(nivel3d))
	vdr = x*vx(nivel3d) + y*vy(nivel3d) + z*vz(nivel3d) 
	write(44,109) nivel3d,vhhh,abs(vz(nivel3d)),esr(nivel3d),
     +          detr(nivel3d)/sumw**3,theta,phi,vm,erms(isrc),
     +          amps(isrc),devwgt(isrc),vdr
 108	format(10a)
 109	format(i4,3f10.1,E15.4,5x,3f8.1,2i5,4f8.1)
 110	format(a4,3a10,a15,5x,3a8,2a5,4a8)
	IF (IS.LT.NSELXED) GOTO 35
C
C.....Finished with all velocities. All selected sources are used. 
	IF (NIV3OVR.GT.0) WRITE(*,101) NIV3OVR, NIND
 101	FORMAT(' INDIVEL3D --',I4,' Excess velocities. Limit =',I6)
	RETURN
	END 
C
C
C  =======================================================================
C
	LOGICAL FUNCTION IND3DOK(VV,RMSERR,DET,RESTART,IS)
C
C	Checks the progress of the INDIVEL3D calculation to assure
C	that all values and errors are within the limits specified in the
C	/SELECT/ block.
C
C	IS is the source index number of the current source, refering
C	to the /SINDEX/ arrays.  NOTE:  IS = NIVEL3D + MINSRC 
C
C..........Parammeters 
$INCLUDE:'COMMON\PARAMS.CMN'
C.....Common Blocks
$INCLUDE:'COMMON\CONTROL.CMN'
$INCLUDE:'COMMON\SELECT.CMN'
$INCLUDE:'COMMON\SINDEX.CMN'
$INCLUDE:'COMMON\SOURCES.CMN'
C
C
	INTEGER IS
	REAL V,VH,EV,VV(3),RMSERR
	REAL*8 DET,TINY
	LOGICAL RESTART
	PARAMETER (TINY = 1.0E-10)
C
	RESTART = .FALSE.
	IND3DOK = .TRUE.
	VH = VV(1)*VV(1) + VV(2)*VV(2)
	V =  SQRT(VH + VV(3)*VV(3))
	VH = SQRT(VH)
	IF (V.GT.0.0) THEN
	   EV = RMSERR/V
	ELSE
	   EV = 100000.
	ENDIF
C
	IND3DOK = (VH.LE.VHMAX).AND.(ABS(VV(3)).LE.VZMAX).AND.
     +      (EV.LE.VERRMAX).AND.(ABS(DET).GT.TINY)
	IF (IND3DOK) RETURN
C
	IF (IS.LE.MINSRC) THEN
C........There is a problem with one or more of the first MINSRC sources.
C........Let some other routine sort this problem out and re-start INDIVEL3D.
	   RESTART = .TRUE.
	   IND3DOK = .FALSE.
	   RETURN
	ELSE
C........It could be that only this last source was a problem.  INDIVEL3D
C........will skip this source.
	   RESTART = .FALSE.
	   IND3DOK = .FALSE.
	ENDIF
C
	RETURN
	END
C
C ====================================================================
C
	SUBROUTINE ZEROSUMS
C	This routine zeros the sums accumulated in the /SUMS/ common block.
C	Normally, this operation is performed whenever INDIVEL3D or INDIVELOS
C	is called, thus isolating one group of measurements from the next.
C	For the sake of future flexibility this process is left as a separate
C	routine and the /SUMS/ are made availible in a common block.
C
C	With the zeroing of /SUMS/ the number Individual velocities NIVEL3D
C	is reset to zero.
C
C..........Parammeters 
$INCLUDE:'COMMON\PARAMS.CMN'
C.....Common Blocks
$INCLUDE:'COMMON\SUMS.CMN'
C
	INTEGER I,J
C
	NSUMS = 0
	SUMW = 0.0
	SUMWVM = 0.0
	SUMWAV = 0.0
	SUMWAA = 0.0
	DO 5 I = 1, 3
	   RVM(I) = 0.0
	   DO 5 J = 1, 3
	      R(I,J) = 0.0
 5	CONTINUE
C
	RETURN
	END
C
C ====================================================================
C
	REAL FUNCTION RMSESQ(V)
C
C	Calculates from the /SUMS/ block the Root mean square error 
C	associated with those sums and their fit to the velocity vector V.
C
C.....Common Blocks
$INCLUDE:'COMMON\SUMS.CMN'
C
	REAL V(3),VX,VY,VZ,HUGE
	PARAMETER (HUGE = 1.0E30)
C
	VX = V(1)
	VY = V(2)
	VZ = V(3)
C
	RMSESQ = VX*VX*R(1,1) + VY*VY*R(2,2) + VZ*VZ*R(3,3) + SUMWVM 
     +   +  2.0*(VX*VY*R(1,2) + VX*VZ*R(1,3) + VY*VZ*R(2,3)) 
     +   -  2.0*(VX*RVM(1)    + VY*RVM(2)    + VZ*RVM(3)   )
C
	IF (SUMW.NE.0.0) THEN
	   RMSESQ = RMSESQ/SUMW
	ELSE
	   RMSESQ = HUGE
	ENDIF
	IF (RMSESQ.LE.0.0)  THEN
	   RMSESQ = 0.0
	ENDIF
	RMSESQ = SQRT(RMSESQ)
	RETURN
	END
C
C ====================================================================
C
	REAL FUNCTION WEIGHT(ISRC,IWEIGH)
C
C	Determine the Least-Squares weight for the source ISRC in the
C	/SOURCES/ block.  IWEIGH is the controling parameter.
C
C.....Parameters
$INCLUDE:'COMMON\PARAMS.CMN'
C.....Common Blocks
$INCLUDE:'COMMON\SOURCES.CMN'
C
	INTEGER ISRC,IWEIGH
	IF (IWEIGH.EQ.1) THEN
C........Log Density
	   WEIGHT = AMPS(ISRC) 
	ELSE IF (IWEIGH.EQ.2) THEN
C........Log Denisty*Velocity Magnitude
	   WEIGHT = AMPS(ISRC)*ABS(VRS(ISRC))
	ELSE IF (IWEIGH.EQ.3) THEN
C........Linear Density 
	   WEIGHT = 10**(AMPS(ISRC)/10.)
	ELSE IF (IWEIGH.EQ.4) THEN
C........Linear Density*Velocity Magnitude.
	   WEIGHT = (10**(AMPS(ISRC)/10.))*ABS(VRS(ISRC))
	ELSE IF (IWEIGH.EQ.5) THEN
C........Velocity Magnitude.
	   WEIGHT = ABS(VRS(ISRC))
	ELSE
C........No Weighting
	   WEIGHT = 1
	ENDIF
	RETURN
	END
C
C =========================================================================
C
	SUBROUTINE ADDTOSUMS(VM,W,X,Y,Z)
C
C	Add to the /SUMS/ arrays the data point (source) at position (X,Y,Z)
C	with velocity Vm and weight W.  (X,Y,Z) is the unit vector for the
C	current source position.
C	Update the variables:
C	   - SUMW which is the sum of the weights W.
C	   - SUMWVM which is the sum of W*Vm^2.
C	   - SUMWAV which is the sum of W*Vm  
C	The matrix structure appears below.
C	
C     |---                                 ---| |-  -|     |---       ---|
C     | sum(W*X*X)   sum(W*X*Y)   sum(W*X*Z)  | | Vx |     | sum(W*X*Vm) |
C     |                                       | |    |     |             |
C     | sum(W*X*Y)   sum(W*Y*Y)   sum(W*Y*Z)  | | Vy |  =  | sum(W*Y*Vm) |
C     |                                       | |    |     |             |
C     | sum(W*X*Z)   sum(W*Y*Z)   sum(W*Z*Z)  | | Vz |     | sum(W*Z*Vm) |
C     |---                                 ---| |-  -|     |---       ---|
C	
C     |---                                  --| |-    -|     |--    --|
C     |    R(1,1)      R(1,2)       R(1,3)    | | V(1) |     | RVM(1) |
C     |                                       | |      |     |        |
C     |    R(2,1)      R(2,2)       R(2,3)    | | V(2) |  =  | RVM(2) |
C     |                                       | |      |     |        |
C     |    R(3,1)      R(3,2)       R(3,3)    | | V(3) |     | RVM(3) |
C     |---                                  --| |-    -|     |--    --|
C
C	For Line-of-Sight velocities, the following are needed:
C
C	       SUMWAV = SUM { W(i)*[Alos.R(i)*Vm] }
C	       SUMWAA = SUM { W(i)*[Alos.R(i)]**2 }           
C	Where:
C	   Alos  = Defined Line-of-Sight direction. Defined in /BOUNDS/
C	   W(i)  = The weights as defined above.
C	   R(i)  = The source position unit vectors. 
C	   Vm    = Measured Source radial velocity.
C
C	*** NOTE ***  A point may be subtracted by supplying a negative W.
C
C.....For /SUMS/  Common Block
$INCLUDE:'COMMON\SUMS.CMN'
C
	REAL VM,W,X,Y,Z,SIGHTLINE,SL
ct	integer i,j
C
	R(1,1) = R(1,1) + W*X*X
	R(1,2) = R(1,2) + W*X*Y
	R(1,3) = R(1,3) + W*X*Z
	R(2,1) = R(1,2)
	R(2,2) = R(2,2) + W*Y*Y
	R(2,3) = R(2,3) + W*Y*Z
	R(3,1) = R(1,3)
	R(3,2) = R(2,3)
	R(3,3) = R(3,3) + W*Z*Z
C
	RVM(1) = RVM(1) + W*X*VM
	RVM(2) = RVM(2) + W*Y*VM
	RVM(3) = RVM(3) + W*Z*VM
C
	SUMW   = SUMW   + W
	SUMWVM = SUMWVM + W*VM*VM
C
	SL = SIGHTLINE(X,Y,Z)
	SUMWAV = SUMWAV + W*SL*VM
	SUMWAA = SUMWAA + W*SL*SL
C
	IF (W.GE.0.0) THEN
	   NSUMS = NSUMS + 1
	ELSE
	   NSUMS = NSUMS - 1
	ENDIF
c
ct	write(44,*) ' ADDTOSUMS -- Nsums,Vm,W,X,Y,Z,Sumw,SumwVm  = '
ct	write(44,102) nsums,vm,z,x,y,z,sumw,sumwvm
ct	write(44,*) ' And the sums are:'
ct	write(44,101) ((r(i,j),j=1,3),rvm(i),i=1,3)
ct 101	format (5x,3f10.2,'  =  ',f10.2/)
ct 102	format (i5,10f10.2)
	RETURN
	END
C
C =========================================================================
C
	SUBROUTINE SOLVESYS(V,DET,SINGULAR)
C
C	Solve the system of equations [R][V] = [RVm] for [V] given 
C	[R] and [RVm] from the /SUMS/ common block.  The method of
C	LU decomposition is used.  See "Numerical Recipies" pp 31-39
C	for further information on LU decomposition.  The routines
C	LUDECOMP and LUBAKSUB are in the \DRIFT\MATH.FOR file.
C	The /SUMS/ remain undisturbed.
C
C.....For /SUMS/  Common Block
$INCLUDE:'COMMON\SUMS.CMN'
C
	INTEGER I,J,INDX(3)
	REAL A(3,3),V(3),D
	REAL*8 DET
	LOGICAL SINGULAR
C
C.....Preserve the values in the R matrix.
	DO 5 I = 1, 3
	   V(I) = RVM(I)
	   DO 5 J = 1, 3
	      A(I,J) = R(I,J)
 5	CONTINUE
	DET = 0.0
C
	CALL LUDECOMP(A,3,3,INDX,D,SINGULAR)
	IF (SINGULAR) RETURN
	CALL LUBAKSUB(A,3,3,INDX,V)
C.....Calculate the Determinant of the A matrix.
	DET = D
	DO 10 I = 1, 3
	   DET = DET * A(I,I)
 10	CONTINUE
	RETURN
	END

