$NOTRUNCATE
$DECLARE
	PROGRAM DOPOFF
C
C	This program calculates the mean and standard deviation of the
C	Doppler offsets in a MAPDATA file.  The output is a text file
C	vs time suitable for plotting.
C
C
C	COMMON Blocks
C
C  /BOUNDS/    -  Data in this common block define the limits in
C                    Frequency and Range for each of the NFRB's.  Range
C	               can be either virtual height or true height.
C  NFRBS          -  Number of FRB's defined.  Must be <= NFRB.
C  BFRQ(NFRB,2)   -  Bounds upon the FReQuency, upper ,1) and lower ,2)
C  BRNG(NFRB,2)   -  Bounds upon the RaNGe, upper ,1) and lower ,2)
C  NBSRC(NFRB)    -  Number of sources in each FRB
C
C  /CONTROL/
C	IN          - Input unit number.
C	IO          - Output unit number.
C	ITIME       - Format for time output.
C	IGROUP      - Grouping method.
C	IWEIGHT     - Weighting method.
C	ISPECT      - Units for spectral output.
C	IOUT        - Output format.
C
C  /DMOMENT/ Central moments, in Doppler.
C	AVE       - Average.
C	ADEV      - Average Deviation.
C	SDEV      - Standard Deviation
C	VAR       - Variance
C	SKEW      - Skewness
C	CURT      - Kurtosis
C	DMED      - Median
C	QUPPER    - Upper Quartile.
C	QLOWER    - Lower Quartile.
C
C   /HISTOGRAM/ - One bin for every 15 min period in the day.
C                 N15 is the number of 15 min bins daily.
C	            NHP is the MAX number of points for each bin.
C	HAVE(N15,NHP)      - Histogram AVErage.
C	HSDEV(N15,NHP)     - Histogram Standard DEViation.
C	HDMED(N15,NHP)     - Histogram Median.
C	HDQU(N15,NHP)      - Histogram Upper Quartile
C	HDQL(N15,NHP)      - Histogram Lower Quartile
C	HPTS(N15)          - Number of Histogram Points for each bin.
C
C  /MAPDATA/   Data read in from input file (SKY output)
C	NMAP      	-  Record number from original tape (useless here)
C	IFF         -  Frequency/height number (1-4)
C	IPREF(89)   -  Digisonde Preface characters
C	FREQ        -  Frequency of measurement MHz
C	RANGE       -  Virtual height of the data, Km
C	GAIN        -  Receiver attenuation, dB
C	MPAMP       -  Most Probable AMPlitude, dB
C	MAXAMP      -  Maximum Amplitude, dB
C	ZMAX        -  Maximum Zenith angle, degrees
C	NUMSRC      -  # of sources that follow
C	YMAP(256)   -  Y coordinates of sources
C	XMAP(256)   -  X coordinates of sources
C	MAPAMP(256) -  Amplitudes of sources, dB
C
C  /SELECT/  Values which effect the selection of CASES, SOURCES, VELOCITIES
C	MINSRC      -  Minimum # of sources to use for calculation
C	MINDOPP     -  Minimum Doppler # to use
C	MAXDOPP     -  Maximum Doppler # to use
C	MPOSERR     -  Maximum positional error to allow (degrees).
C	MCASESNR    -  Subcase maximum Signal - Noise case selection threshold.
C	MINLOBE     -  Lower threshold for source selection, realtive to
C	               subcase maximum amplitude.
C	MSRCSNR     -  Minimum SNR for each source.
C
C  /SINDEX/  Index arrays for active (selected) /SOURCES/
C	NSELXED       - Number of active entries in SELINDX index.
C	SELINDX(NIND) - Index of those sources selected for INDIVEL2.
C	                Contains location of source in /SOURCES/
C
C  /SOURCES/ Information on each source required for determining the
C	       velocity vector.  NSS is the parameter for the maximum
C            number of saved sources.
C	NSRCRS     - Current number of sources  (INTEGER*4)
C	THETAS(NSS) - INTEGER*2 arrival elevation angle*10, from Zenith.
C	PHIS(NSS)   - INTEGER*2 arrival azimuth*10, from magnetic North, 
C	              increasing to the EAST, like Navigational coordiantes.
C	              *** NOTE ***  This is opposite SKYMAP data conventions. 
C	ERMS(NSS)   - INTEGER*1 source RMS positional error.
C	AMPS(NSS)   - INTEGER*1 amplitude of the source,dB.
C	NOIS(NSS)   - INTEGER*1 noise (MPA) of the source,dB.
C	VRS(NSS)    - Radial velocity of the source. (REAL*4).
C	DOPS(NSS)   - Doppler shift (Hz) for this source. (REAL*4).
C	IDBINS(NSS) - Integer Doppler bin number (INTEGER*2)
C	BINS(NSS)   - INTEGER*2 The /BOUNDARY/ bin for this source.
C	ACTIVE(NSS) - LOGICAL*1 flag indicating selection of this source.
C
C  /TIME/
C	STARTTIME, ENDTIME, NOWTIME, LASTTIME
C	ATIME       - Data Accumulation time = time over which data is
C	              averaged/smoothed. (seconds)
C	BTIME       - An alternate ATIME, for unspecified use.
C	ACCSTIME    - The accumulation (averaging) start time. 
C	CSTIME,CETIME,CNTIME,CCSTIME,CLTIME - Times in CHARACTER*17 format
C	                         YYYY DDD HH:MM:SS 
C	NR2DO       - The number of Subcases to process if ENDTIME unspecified.
C
C
$INCLUDE:'\DRIFT\PLAY\PARAMS.CMN'
$INCLUDE:'\DRIFT\PLAY\CONTROL.CMN'
$INCLUDE:'\DRIFT\PLAY\DMOMENT.CMN'
$INCLUDE:'\DRIFT\PLAY\BOUNDS.CMN'
$INCLUDE:'\DRIFT\PLAY\HISTO.CMN'
$INCLUDE:'\DRIFT\PLAY\MAPDATA.CMN'
$INCLUDE:'\DRIFT\PLAY\SELECT.CMN'
$INCLUDE:'\DRIFT\PLAY\SINDEX.CMN'
$INCLUDE:'\DRIFT\PLAY\SOURCES.CMN'
$INCLUDE:'\DRIFT\PLAY\TIME.CMN'

C
C	PROGRAM VARIABLES
	INTEGER I,IBIN1,NSKIP,NSUBCASE,PARTITION
	REAL PGOOD
	LOGICAL EOF,CALCVEL,TIMEBREAK
	REAL*8 PARSETIME
	CHARACTER*17 CTIMELTT
C
	CALL INPUTDATA()
C.....Start searching for the first record desired.
	NSKIP = 0
	IF (STARTTIME.EQ.0) THEN
	   WRITE(*,*) ' Starting at begining of file ...'
	   CALL READMAPDATA(IN,EOF)
	   LASTTIME = 0.0
	   NOWTIME = PARSETIME(IPREF)
	   LASTTIME = NOWTIME
	   CNTIME = CTIMELTT(IPREF)
	   CLTIME = CNTIME
	ELSE
 11	   CALL READMAPDATA(IN,EOF)
	   IF(EOF) GO TO 1450
	   LASTTIME = NOWTIME
	   CLTIME = CNTIME
	   NOWTIME = PARSETIME(IPREF)
	   CNTIME = CTIMELTT(IPREF)
	   IF (NOWTIME.LT.STARTTIME) THEN
	      WRITE(*,'(1X,2A)') CNTIME,' Skipping ...'
	      NSKIP = NSKIP + 1
	      GOTO 11
	   ENDIF
	   WRITE(*,133) CNTIME,NSKIP
	ENDIF
C
 133	FORMAT(' Found desired start time ',A20,'.  Skipped ',I4,
     +         ' Subcases.')
C
C.....Desired record has been found, start processing.
C.....Write header out to unit IO.
	CALL HEADER(IO)
C
C	The sequence of data processing is to fill up the /SOURCES/ common
C	block with data for the interval specified in the /CONTROL/ block.
C	After all sources have been found, desired info calculated.
C
	NSUBCASE = NSKIP + 1
C.....Date and time update, zero sources and prepare for calculations.
	CALL INITIALIZE()
C
C.....Start the loop which accumulates sources for the  desired period
C.....of time.
C
 1	CONTINUE
C
C........Determine if it is time to calculate a GROUP velocity.
	   CALCVEL = TIMEBREAK()
	   IF (CALCVEL) THEN
C...........Do all the calculations
	      DO 33 I = 1, NFRBS
	         CALL CHOOSE(I)
	         CALL DOPWIDTH()
	         CALL DOPOUT(NSELXED)
	         CALL SETHIST()
 33	      CONTINUE

C...........Initialize things for the next calculation.
	      CALL INITIALIZE()
	   ENDIF
C
C........Continue with source accumulation
	   CALL FILTER(PGOOD)
	   IBIN1 = PARTITION()
	   CALL SAVESOURCE(IBIN1)
	   WRITE(*,134) CNTIME,NSUBCASE,PGOOD
 134	   FORMAT(1X,A17,', subcase # ',I5,3X,F4.0'% good sources.')
C
	   CALL READMAPDATA(IN,EOF)
	   NSUBCASE = NSUBCASE + 1
	   LASTTIME = NOWTIME
	   CLTIME = CNTIME
	   NOWTIME = PARSETIME(IPREF)
	   CNTIME = CTIMELTT(IPREF)
	   IF (EOF) GOTO 1450
	IF ((NOWTIME.LE.ENDTIME).AND.(NSUBCASE.LE.NR2DO)) GOTO 1
C
 1450	CONTINUE
C.....Finalize calculations on any data waiting to be processed.
	CALL HISTOUT()
	WRITE (*,*) ' Terminating after ',NSUBCASE-1, ' Subcases.'
	CLOSE (UNIT=IO)
	CLOSE (UNIT=IN)
C
	IF (EOF) WRITE(*,*) ' End of data from input file '
	END
C
C  =======================================================================
C
	SUBROUTINE HISTOUT()
C
C	Outputs the /HISTO/ data to the file HISTO.OUT
C
$INCLUDE:'\DRIFT\PLAY\PARAMS.CMN'
$INCLUDE:'\DRIFT\PLAY\CONTROL.CMN'
$INCLUDE:'\DRIFT\PLAY\HISTO.CMN'
$INCLUDE:'\DRIFT\PLAY\TIME.CMN'
C
C
	INTEGER I,IO2,IM,IH,NM,IROUND,IBB,NPT
	REAL R,DHOUR,DATA(NSS),AVE,ADEV,SDEV,VAR,SKEW,CURT,
     +     DMED,QUPPER,QLOWER,TOSS
	CHARACTER*20 INF
C
	IO2 = 22
	INF = 'HISTO.OUT'
C
	OPEN(UNIT=IO2,FILE=INF,FORM='FORMATTED',
     +     ACCESS='SEQUENTIAL',STATUS='UNKNOWN')
C
	NM = IROUND(60./BTIME)
	NM = MAX(NM,1)
C
C.....Averages.
C	WRITE(IO2,*) ' Averages'
C	CALL HEADER(IO2)
C	DO 20 IH = 0, 23
C	DO 20 IM = 0,NM-1
C	   DHOUR = IH + BTIME*IM/60.
C	   IBB = NM*IH + IM + 1
C	   IBB = MIN(IBB,N15)
C	   NPT = HPTS(IBB)
C	   DO 10 I = 1, NPT
C	      DATA(I) = HAVE(IBB,I)
C 10	   CONTINUE
C	   CALL MOMENT(DATA,NPT,AVE,ADEV,SDEV,VAR,SKEW,CURT)
C	   CALL MEDIAN10(DATA,NPT,DMED,QUPPER,QLOWER)
CC
C	   IF (IOUT.EQ.1) THEN
C	      WRITE(IO2,101) DHOUR,AVE,SDEV,DMED,QUPPER,QLOWER,
C     +                     FLOAT(NPT)
C	   ELSE
C	      WRITE(IO2,101) DHOUR,AVE+SDEV,AVE-SDEV,AVE,
C     +                    QUPPER,QLOWER,DMED,FLOAT(NPT)
C	   ENDIF
C 20	CONTINUE
 101	FORMAT (1X,12F10.4)
 102	FORMAT (1X,12A10)
C	WRITE(IO2,*) ' '
C
C.....Medians.
	WRITE(IO2,*) ' Medians'
	WRITE(IO2,102) 'Hour','Qupper','Qlower','Median','Ave+Std',
     +   'Ave-Std','Average','Npoints'
C	CALL HEADER(IO2)
	DO 21 IH = 0, 23
	DO 21 IM = 0,NM-1
	   DHOUR = IH + BTIME*IM/60.
	   IBB = NM*IH + IM + 1
	   IBB = MIN(IBB,N15)
	   NPT = HPTS(IBB)
C........Middle
	   DO 11 I = 1, NPT
	      DATA(I) = HDMED(IBB,I)
	      CALL MOMENT(DATA,NPT,AVE,ADEV,SDEV,VAR,SKEW,CURT)
 11	   CONTINUE
	   CALL MEDIAN1(DATA,NPT,DMED)
C........Upper
	   DO 12 I = 1, NPT
	      DATA(I) = HDQU(IBB,I)
 12	   CONTINUE
	   CALL MEDIAN1(DATA,NPT,QUPPER)
C........Lower
	   DO 13 I = 1, NPT
	      DATA(I) = HDQL(IBB,I)
 13	   CONTINUE
	   CALL MEDIAN1(DATA,NPT,QLOWER)
	   WRITE(IO2,101) DHOUR,QUPPER,QLOWER,DMED,
     +                       AVE+SDEV,AVE-SDEV,AVE,FLOAT(NPT)
 21	CONTINUE
	CLOSE(UNIT=IO)
	WRITE(*,*)' Histogram file ',INF,' created.'
	RETURN
	END
C
C  =======================================================================
C
	SUBROUTINE SETHIST()
C
C	Place data in the /HISTOGRAM/ block.
C
$INCLUDE:'\DRIFT\PLAY\PARAMS.CMN'
$INCLUDE:'\DRIFT\PLAY\DMOMENT.CMN'
$INCLUDE:'\DRIFT\PLAY\HISTO.CMN'
$INCLUDE:'\DRIFT\PLAY\TIME.CMN'
C
	CHARACTER*17 CTT,CTIME
	INTEGER IBB,NPT,NM,IROUND,IH,IM
	REAL R
C
C.....Determine which bin to use.
C.....NM is the Number of Minute blocks in an hour.
C
	NM = IROUND(60./BTIME)
	NM = MAX(NM,1)
	CTT = CTIME(5)
	READ(CTT,101) IH,IM
	IM = INT(REAL(IM/BTIME))
	IBB = NM*IH + IM + 1
	IBB = MIN(IBB,N15)
	NPT = HPTS(IBB)
	NPT = NPT + 1
	NPT = MIN(NPT,NHP)
	HAVE(IBB,NPT) = AVE
	HSDEV(IBB,NPT) = SDEV
	HDMED(IBB,NPT) = DMED
	HDQU(IBB,NPT) = QUPPER
	HDQL(IBB,NPT) = QLOWER
	HPTS(IBB) = NPT
C
 101	FORMAT(9X,I2,1X,I2)
	RETURN
	END
C

C  =======================================================================
C
	SUBROUTINE DOPWIDTH()
C
C	This routine calculates the mean Doppler and the Doppler spread
C	for the /SELECT/ed /SOURCES/
C
$INCLUDE:'\DRIFT\PLAY\PARAMS.CMN'
$INCLUDE:'\DRIFT\PLAY\CONTROL.CMN'
$INCLUDE:'\DRIFT\PLAY\DMOMENT.CMN'
$INCLUDE:'\DRIFT\PLAY\SINDEX.CMN'
$INCLUDE:'\DRIFT\PLAY\SOURCES.CMN'
C
	INTEGER ISRC,KNS
	REAL DATA(NSS),W(NSS)
C
	IF (NSELXED.LT.1) RETURN
C
	KNS = MIN(NSELXED,NSS)
	DO 10 ISRC = 1, NSELXED
	   IF (ISPECT.EQ.1) THEN
	      DATA(ISRC) = DOPS(ISRC)
	   ELSE IF (ISPECT.EQ.2) THEN
	      DATA(ISRC) = 1000.*DOPS(ISRC)
	   ELSE IF (ISPECT.EQ.3) THEN
	      DATA(ISRC) = FLOAT(IDBINS(ISRC))
	   ELSE
	      DATA(ISRC) = VRS(ISRC)
	   ENDIF
C
	   W(ISRC) = AMPS(ISRC)
 10	CONTINUE
	IF (IWEIGHT.EQ.1) THEN
	   CALL MOMENT(DATA,KNS,AVE,ADEV,SDEV,VAR,SKEW,CURT)
	ELSE IF (IWEIGHT.EQ.2) THEN
	   CALL WMOMENT(DATA,W,KNS,AVE,ADEV,SDEV,VAR,SKEW,CURT)
	ENDIF
	CALL MEDIAN10(DATA,KNS,DMED,QUPPER,QLOWER)
	RETURN
	END
C
C  =======================================================================
C
	SUBROUTINE DOPOUT(NNN)
C
C	Write to the output file the time and Doppler statistics.
C	NNN is the number of sources.
c
$INCLUDE:'\DRIFT\PLAY\PARAMS.CMN'
$INCLUDE:'\DRIFT\PLAY\CONTROL.CMN'
$INCLUDE:'\DRIFT\PLAY\DMOMENT.CMN'
C
C
	INTEGER*2 NNN,KD,I
	CHARACTER*17 CTIME,CTT
	CHARACTER*10 LINE(20)
C
	IF (NNN.LT.1) RETURN
	IF (ITIME.EQ.1) THEN
	   CTT = CTIME(2)
	ELSE IF (ITIME.EQ.2) THEN
	   CTT = CTIME(6)
	ELSE IF (ITIME.EQ.3) THEN
	   CTT = CTIME(7)
C........Get rid of the year
	   WRITE(CTT,'(11A,6X)') CTT(5:17)
	ENDIF
C
	IF (IOUT.EQ.1) THEN
	   WRITE(IO,101) CTT,AVE,SDEV,DMED,QUPPER,QLOWER,FLOAT(NNN)
	ELSE
	   WRITE(IO,101) CTT,AVE+SDEV,AVE-SDEV,AVE,
     +                       QUPPER,QLOWER,DMED,FLOAT(NNN)
	ENDIF

C
 101	FORMAT (1X,A17,12F10.3)
	RETURN
	END
C
C  =======================================================================
C
	SUBROUTINE HEADER(III)
C
C	Send out appropriate header for data printed.
C
$INCLUDE:'\DRIFT\PLAY\CONTROL.CMN'
C
	INTEGER III
C
	IF (IOUT.EQ.1) THEN
	   WRITE(III,101) 'Time','Ave','Std Dev','Median','Qu','Ql',
     +                 '#Src'
	ELSE
	   WRITE(III,101) 'Time','Ave+Sdev','Ave-Sdev','Ave','Qhigh',
     +                 'Qlow','Median','#Src'
	ENDIF
 101	FORMAT (A17,12A10)
	RETURN
	END
C
C  =======================================================================
C
	SUBROUTINE CHOOSE(IBIN)
C
C	Select those sources from /SOURCES/ which are to be used the
C	INDIVEL2 routine to be translated into a vector velocity. Usually
C	that means all sources with a BINS# = IBIN
C	While selecting sources, build an index of selected sources for
C	quick reference into the large /SORUCES/ array.
C	A re-ordering of source sequence, if necessary, is appropriate here.
C	The /CONTROL/ parameter ICHOOSE can influence the operation of this
C	routine if necessary.
C
C.....Parameters
$INCLUDE: '\DRIFT\PLAY\PARAMS.CMN'
C.....Common Blocks
$INCLUDE: '\DRIFT\PLAY\SOURCES.CMN'
$INCLUDE: '\DRIFT\PLAY\CONTROL.CMN'
$INCLUDE: '\DRIFT\PLAY\SINDEX.CMN'

	INTEGER IBIN,I
C
	NSELXED = 0
	DO 10 I = 1, NSRCRS
	   IF(BINS(I).EQ.IBIN) THEN
	      ACTIVE(I) = .TRUE.
	      NSELXED = NSELXED + 1
	      SELINDX(NSELXED) = I
	   ENDIF
 10	CONTINUE
	RETURN
	END
C
C  =======================================================================
C
	SUBROUTINE FILTER(PGOOD)
C
C	Apply the criteria in the /SELECT/ block to decide which /MAPDATA/
C	sources to keep and which ones to discard.
C	Deleted sources have a Negative MAPRMS.
C
$INCLUDE: '\DRIFT\PLAY\MAPDATA.CMN'
$INCLUDE: '\DRIFT\PLAY\SELECT.CMN'
C
C
	INTEGER I,MD,NTOSS,MDMAX,MDOPLINE
	REAL PGOOD
	LOGICAL AOK
C
C	Check ratio of the subcase maximum amplitude to the subcase noise
C	to see if the case is to be kept.  If not, toss all sources.
C	PGOOD is the Percentage of GOOD sources in this batch.
C
	IF (((MAXAMP-MPAMP).LT.MCASESNR).OR.NUMSRC.EQ.0) THEN
	   NTOSS = NUMSRC
	   PGOOD = 0.0
	   NUMSRC = 0
	   RETURN
	ENDIF
C.....Passed the subcase test, now check each source.
	NTOSS = 0
	MDMAX = MDOPLINE()
	IF (MAXDOPP.LT.0) THEN
	   MDMAX = MDMAX - ABS(MAXDOPP)
	ELSE
	   MDMAX = ABS(MAXDOPP)
	ENDIF
C
	DO 20 I = 1, NUMSRC
	   MD = ABS(MAPDOP(I))
	   AOK = (MD.GE.MINDOPP).AND.(MD.LE.MDMAX).AND.
     +         ((MAPAMP(I)-MPAMP).GE.MSRCSNR).AND.
     +         (MAPRMS(I).LE.MPOSERR).AND.
     +         ((MAXAMP-MAPAMP(I)).LE.MINLOBE)
	   IF (.NOT.AOK) THEN
	      NTOSS = NTOSS + 1
	      MAPRMS(I) = -MAPRMS(I)
	   ENDIF
 20	CONTINUE
	PGOOD = 100.0 - (100.*NTOSS)/NUMSRC
	RETURN
	END
C
C
C  =======================================================================
C
	SUBROUTINE INITIALIZE()
C
C	Resets all necessary variables between calculations.
C
C.....Parameters
$INCLUDE: '\DRIFT\PLAY\PARAMS.CMN'
C.....Common Blocks
$INCLUDE: '\DRIFT\PLAY\BOUNDS.CMN'
$INCLUDE: '\DRIFT\PLAY\CONTROL.CMN'
$INCLUDE: '\DRIFT\PLAY\DMOMENT.CMN'
$INCLUDE: '\DRIFT\PLAY\SOURCES.CMN'
$INCLUDE: '\DRIFT\PLAY\TIME.CMN'
C
	INTEGER I
	LOGICAL ROUND
	CHARACTER*17 CTIME
	REAL*8 PARCTIME
C.....No more sources.
	NSRCRS = 0
	DO 10 I = 1, NFRB
	   NBSRC(I) = 0
 10	CONTINUE
	AVE = 0.0
	ADEV = 0.0
	SDEV = 0.0
	VAR  = 0.0
	SKEW = 0.0
	CURT = 0.0
	DMED = 0.0
	QUPPER = 0.0
	QLOWER = 0.0
C
	ACCSTIME = NOWTIME
	CCSTIME = CNTIME
	RETURN
	END
C
C  =======================================================================
C
	INTEGER FUNCTION PARTITION()
C
C	Assign a PARTITION number to the data in the /MAPDATA/ block.
C	Use information in the /BOUNDS/ block to make this judgment.
C
C.....Parameters
$INCLUDE: '\DRIFT\PLAY\PARAMS.CMN'
C.....Common Blocks
$INCLUDE: '\DRIFT\PLAY\MAPDATA.CMN'
$INCLUDE: '\DRIFT\PLAY\BOUNDS.CMN'
C
C
	INTEGER IBIN
	REAL RNG,FRM
	LOGICAL THISFBIN,THISHBIN
C
	PARTITION = 0
	DO 10 IBIN = 1, NFRB
	   RNG = RANGE/1000.
	   FRM = FREQ/1.0E6
	   THISHBIN = (RNG.GE.BRNG(IBIN,1)).AND.(RNG.LE.BRNG(IBIN,2))
	   THISFBIN = (FRM.GE.BFRQ(IBIN,1)).AND.(FRM.LE.BFRQ(IBIN,2))
	   IF (THISFBIN.AND.THISHBIN) PARTITION = IBIN
 10	CONTINUE
	IF (PARTITION.EQ.0) WRITE(*,101) FREQ/1.E6,RANGE/1000.
 101	FORMAT(' PARTITION -- F =',F6.2,' MHz R =',F5.0,' Km not found.')
	RETURN
	END
C
C  =======================================================================
C
	SUBROUTINE SAVESOURCE(IBIN)
C
C	This routine transfers the necessary data from the /MAPDATA/ block
C	to the /SOURCES/ block with appropriate translation.
C	   IBIN  - The bin number to which the data in this spectrum has
C	           been assigned by the PARTITION function.
C
C
C.....Parameters.
$INCLUDE: '\DRIFT\PLAY\PARAMS.CMN'
C
C.....Common Blocks
$INCLUDE:'\DRIFT\PLAY\MAPDATA.CMN'
$INCLUDE:'\DRIFT\PLAY\SOURCES.CMN'
$INCLUDE:'\DRIFT\PLAY\BOUNDS.CMN'
C
C
	INTEGER I,IBIN,IROUND,NOVER
	REAL DFR,DOPRES,DOPFRQ,C
	REAL ASIND,SIND,ATAND2
C
	DATA C /2.997924574E+8/
C
	IF ((IBIN.LT.1).OR.(IBIN.GT.NFRB).OR.(NUMSRC.LT.1)) RETURN
C
	DFR = DOPRES()
	DO 30 I = 1, NUMSRC
C........Negative MAPRMS means do not save this source.
	   IF (MAPRMS(I).GE.0) THEN
	      NSRCRS = NSRCRS + 1
C...........Array boundary checking.
	      IF (NSRCRS.GT.NSS) THEN
	         NSRCRS = NSS
	         NOVER = NOVER + 1
	         WRITE(*,101) NOVER,NSS
	         RETURN
	      ENDIF
	      NOVER = 0
	      THETAS(NSRCRS) = IROUND(ASIND(SQRT(XMAP(I)*XMAP(I) + 
     +            YMAP(I)*YMAP(I))/20.0 * SIND(ZMAX))* 10.0)
	      PHIS(NSRCRS) = IROUND(ATAND2(-YMAP(I),XMAP(I)) * 10.0)
	      AMPS(NSRCRS) = MAPAMP(I) + GAIN
	      NOIS(NSRCRS) = MPAMP + GAIN
	      ERMS(NSRCRS) = MAPRMS(I)
	      IDBINS(NSRCRS) = MAPDOP(I)
	      IF (MAPDOP(I).GT.0) THEN
	         DOPFRQ = DFR*(FLOAT(MAPDOP(I)) - 0.5)
	      ELSE
	         DOPFRQ = DFR*(FLOAT(MAPDOP(I)) + 0.5)
	      ENDIF
	      DOPS(NSRCRS) = DOPFRQ
	      IF (FREQ.NE.0.0) THEN
	         VRS(NSRCRS) = -C*DOPFRQ/(2.0*FREQ)
	      ELSE
	         VRS(NSRCRS) = 0.0
	      ENDIF
	      BINS(NSRCRS) = IBIN
	      ACTIVE(NSRCRS) = .FALSE.
C...........Now indicate to the addition of this source to IBIN
	      NBSRC(IBIN) = NBSRC(IBIN) + 1
	   ENDIF
 30	CONTINUE
C
 101	FORMAT(' SAVESOURCE --',I4,' Excess sources.  Limit =',I6)
	RETURN
	END
C
C  =======================================================================
C
	REAL FUNCTION DOPRES()
C
C	Determines from the preface IPREF and the sounding frequency FREQ
C	the Doppler Frequency Resolution, which can be used to translate the
C	Doppler numbers in /MAPDATA/ to frequencies.
C	Ex:  Given a Doppler # I and the correct value of DOPRES then:
C			Doppler Shift (Hz) = DOPRES*(I - 0.5)  I > 0
C			Doppler Shift (Hz) = DOPRES*(I + 0.5)  I < 0
C	                *** for a Digisonde 256.  
C
C	                    Common Blocks
C.....For /MAPDATA/
$INCLUDE: '\DRIFT\PLAY\MAPDATA.CMN'
C
	INTEGER*1 NFREQS(16),NANTS(16),NHITES(16),IL,IT3,IN
	INTEGER IREP,NSMPLS,NPULSES
	REAL TINTEG
C
C.......Value =   0 1 2 3 4 5 6 7 8 9 A B C D E F
	DATA NFREQS/1,2,1,2,1,2,1,2,2,4,2,4,2,1,2,4/ 
	DATA  NANTS/1,4,4,8,4,8,1,4,1,4,4,8,4,8,1,4/
	DATA NHITES/2,1,2,1,2,2,8,1,2,1,2,1,2,2,8,1/
C 
	IL = IPREF(45)
	IT3 = IAND(IPREF(47),8)
	IN = IPREF(48)
	IREP = 50*2**((IPREF(49)+1)/2) 
	NSMPLS = 32*(2**IAND(IN,3))
	NPULSES = NANTS(IL+1)*NFREQS(IL+1)*NHITES(IL+1)
	TINTEG = FLOAT(NSMPLS*NPULSES)/FLOAT(IREP)
	DOPRES = (1+IT3)/TINTEG
	RETURN
	END
C
C  =======================================================================
C
	INTEGER FUNCTION MDOPLINE()
C
C	Determines from the preface IPREF the Maximum Absolute Doppler
C	Frequency Number, which can be used to determine if the
C	Doppler numbers in /MAPDATA/ are near the end of the spectrum.
C
C	                    Common Blocks
C.....For /MAPDATA/
$INCLUDE: '\DRIFT\PLAY\MAPDATA.CMN'
C
	INTEGER IN
C
	IN = IPREF(48)
	MDOPLINE = 16*(2**IAND(IN,3))
	RETURN
	END
C
C  =======================================================================
C
	LOGICAL FUNCTION TIMEBREAK()
C
C	Function to determine the time at which to stop accumulating
C	sources and calculate group velocities.
C
C.....Parameters
$INCLUDE:'\DRIFT\PLAY\PARAMS.CMN'
C	                    Common Blocks
$INCLUDE: '\DRIFT\PLAY\CONTROL.CMN'
$INCLUDE: '\DRIFT\PLAY\TIME.CMN'
C
C
	TIMEBREAK = .FALSE.
	IF (IGROUP.EQ.1) THEN
	   TIMEBREAK = .TRUE.
	ELSE IF ((IGROUP.EQ.2).AND.(NOWTIME.NE.LASTTIME)) THEN
C........Case velocities
	   TIMEBREAK = .TRUE.
	ELSE IF ((IGROUP.EQ.3).AND.((NOWTIME-LASTTIME).GT.59)) THEN
C........Group velocities
	   TIMEBREAK = .TRUE.
	ELSE IF ((IGROUP.EQ.4).AND.((NOWTIME-ACCSTIME).GT.ATIME)) THEN
C........Specified interval velocities
	   TIMEBREAK = .TRUE.
	ELSE IF ((IGROUP.EQ.5).AND.
     +                  ((NOWTIME-ACCSTIME).GT.60.*BTIME)) THEN
C........5 Min Histograms.
	   TIMEBREAK = .TRUE.
	ENDIF
	RETURN
	END
C
C ==========================================================================
C 
	SUBROUTINE INPUTDATA()
C
C	Use menus to read in pertinent input parameters for the drift
C	velocity vector calculations.
C
C.....Paramteters
$INCLUDE:'\DRIFT\PLAY\PARAMS.CMN'
C.....Common Blocks
$INCLUDE:'\DRIFT\PLAY\CONTROL.CMN'
$INCLUDE:'\DRIFT\PLAY\SELECT.CMN'
$INCLUDE:'\DRIFT\PLAY\TIME.CMN'
C
C	Variable declaration
	INTEGER GETINT,IJUNK,IROUND,ICFILE
	INTEGER*2 LTT(11)
	CHARACTER*17 CTT,CTIME
	LOGICAL GRAPH,PROBLEM,REPORT,LIST,FILE
	REAL*8 PARSETIME,PARCTIME
	CHARACTER*30 CFILE,LFILE,INFILE
C
C....READ INPUT PARAMETERS, from either the user or a file. 
C
	IO = 50
	IN = 10
C
	WRITE(*,*) ' Skymap Data Manipulator.'
	WRITE(*,*) ' '
C
C.....Input options from the user.
C
 902	WRITE(*,*) 'Enter the name of the file with the source data:'
	READ(*,'(A)') INFILE
	OPEN(UNIT=IN,FILE=INFILE,FORM='FORMATTED',
     +     ACCESS='SEQUENTIAL',STATUS='OLD',ERR=902)

	WRITE(*,*) ' Data Listing Destination:'
	WRITE(*,*) '      1) List to the printer.'
	WRITE(*,*) '      2) List to a file.'
	FILE = (GETINT(1,2).EQ.2)
	
	IF (FILE) THEN
 903	   WRITE(*,*) ' Name of the file for the data listings.'
	   READ(*,'(A)') LFILE
	   OPEN(UNIT=IO,FILE=LFILE,FORM='FORMATTED',MODE='WRITE',
     +       ACCESS='SEQUENTIAL',ERR=903)
	ELSE
	   OPEN(UNIT=IO,FILE='PRN',FORM='FORMATTED',MODE='WRITE',
     +        ACCESS='SEQUENTIAL')
	ENDIF
C
C
C.....Get the input variables.
	WRITE(*,*) ' Time format:'
	WRITE(*,*) '     1) Character*17 format.'
	WRITE(*,*) '     2) Day and Decimal Hour.'
	WRITE(*,*) '     3) Decimal day.'
	ITIME = GETINT(1,3)
C
	WRITE(*,*) ' Print spectral information in:'
	WRITE(*,*) '     1) Hertz.'
	WRITE(*,*) '     2) MiliHertz.'
	WRITE(*,*) '     3) Doppler Bin #.'
	ISPECT = GETINT(1,3)
C
	WRITE(*,*) ' Output Format:'
	WRITE(*,*) '     1) Moments and medians.'
	WRITE(*,*) '     2) High-Low-Middle format.'
	IOUT = GETINT(1,2) 
C
	WRITE(*,*) ' Data grouping options:'
	WRITE(*,*) '     1) Subcase Velocities.'
	WRITE(*,*) '     2) Case Velocities.'
	WRITE(*,*) '     3) Group Velocities.'
	WRITE(*,*) '     4) Specify velocity grouping interval.'
	IGROUP = GETINT(1,4)
	IF (IGROUP.EQ.4) THEN
	   WRITE(*,*) ' Enter number of minutes between velocity',
     +              ' calculations.'
	   READ(*,*) ATIME
	   ATIME = ATIME*60.0
	ENDIF
C
	WRITE(*,*) ' Histogram Options -- 15 Minute bins.'
	WRITE(*,*) '     1) As grouped above.'
	WRITE(*,*) '     2) 15 Minute group.'
	IJUNK = GETINT(1,2)
	BTIME = 15.0
	IF (IJUNK.EQ.2) IGROUP = 5
	IF (ATIME.EQ.0.0) ATIME = 60.*BTIME
C
	WRITE(*,*) ' Spectral line weighting options.'
	WRITE(*,*) '     1) Equal weight.'
	WRITE(*,*) '     2) Amplitude weight.'
	IWEIGHT = GETINT(1,2)
C
C.....Starting and ending times.
C
	NR2DO = 0
 21 	WRITE(*,*) ' Enter starting date and time in the format:'
	WRITE(*,*) ' 1987 022 01:05:02   or enter 0 for first record'
	WRITE(*,*) ' of the file.'
	READ (*,29) CTT
 29	FORMAT (A17)
	STARTTIME = PARCTIME(CTT)
	CSTIME = CTT
	WRITE(*,*) ' Enter ending date and time (same format) or'
	WRITE(*,*) ' enter 0 to select a number of cases.'
	READ(*,29) CTT
	ENDTIME = PARCTIME(CTT)
	CETIME = CTT
	IF (STARTTIME.GT.ENDTIME) THEN
	   WRITE(*,*)' Entered start time exceeds entered ending time.'
	   GOTO 21
	ENDIF
	IF (ENDTIME.EQ.0.0) THEN
	   ENDTIME = 1.0E27
	   WRITE(*,*) ' Enter number of DRIFT sub-cases to process or'
	   WRITE(*,*) ' enter 0 for all records up to EOF.'
	   READ(*,*) NR2DO
	ENDIF
	IF (NR2DO.EQ.0) NR2DO = 9999999

C.....Weight to apply to each data point in the velocity calculation.
C 
C	Source selection criteria.  Used in various locations
C	to toss out 'bad' data points.
C
C	WRITE(*,*) '       Criteria for Source selection:' 
	WRITE(*,*)' Enter the Minimum Absolute Doppler Number (0=ALL).'
	READ(*,*) MINDOPP 
	WRITE(*,*)' Enter the Maximum Absolute Doppler Number (0=ALL).'
	WRITE(*,*)'   Negative values are below spectral limit.'
	READ(*,*) MAXDOPP
	IF (MAXDOPP.EQ.0) MAXDOPP = 9999
	WRITE(*,*) ' Enter Maximum Positional Error (Degrees; 0=ALL)'
	READ(*,*) MPOSERR
	IF (MPOSERR.LE.0) MPOSERR = 9999
	WRITE(*,*) ' Enter SNR threshold for Subcase selection,'
	WRITE(*,*) ' (Subcase Maximum Amplitude - Noise, dB; 0=ALL)'
	READ(*,*) MCASESNR 
	WRITE(*,*) ' Process sources how far below the Subcase peak ?',
     +           ' (dB; 0=ALL)' 
	READ(*,*) MINLOBE
	IF (MINLOBE.LE.0) MINLOBE = 100 
	WRITE(*,*) ' Source Minimum SNR (dB; 0=ALL)'
	READ(*,*) MSRCSNR

C.....Set the BOUNDS for data grouping.
	CALL SETBOUNDS()
	RETURN
	END
C
C
C

C 
C	Functions and things for the VELH program.
C
	REAL FUNCTION SIND(ANGLE)
	REAL ANGLE,RAD
	DATA RAD /0.017453293/
	SIND = SIN(RAD*ANGLE)
	RETURN
	END
C
	REAL FUNCTION COSD(ANGLE)
	REAL ANGLE,RAD
	DATA RAD /0.017453293/
	COSD = COS(RAD*ANGLE)
	RETURN
	END
C
	REAL FUNCTION TAND(ANGLE)
	REAL ANGLE,RAD
	DATA RAD /0.017453293/
	TAND = TAN(RAD*ANGLE)
	RETURN
	END
C
C.....Inverse Trig functions return angles in degrees from 0-360 
	REAL FUNCTION ACOSD(VALUE)
	REAL VALUE,DEG
	DATA DEG /57.29577951/
	IF (ABS(VALUE).GT.1.0) VALUE = SIGN(1.0,VALUE)
	ACOSD = DEG*ACOS(VALUE)
	IF (ACOSD.LT.0.0) ACOSD = ACOSD + 360.0
	RETURN
	END
C
	REAL FUNCTION ASIND(VALUE)
	REAL VALUE,DEG
	DATA DEG /57.29577951/
	IF (ABS(VALUE).GT.1.0) VALUE = SIGN(1.0,VALUE)
	ASIND = DEG*ASIN(VALUE)
	IF (ASIND.LT.0.0) ASIND = ASIND + 360.0
	RETURN
	END
C
	REAL FUNCTION ATAND2(VALUE1,VALUE2)
	REAL VALUE1,VALUE2,DEG
	DATA DEG /57.29577951/
	ATAND2 = 0.0
	IF ((VALUE1.EQ.0).AND.(VALUE2.EQ.0)) RETURN
	ATAND2 = DEG*ATAN2(VALUE1,VALUE2)
	IF (ATAND2.LT.0.0) ATAND2 = ATAND2 + 360.0
	RETURN
	END
C
C
	SUBROUTINE CARTESIAN(R,THETA,PHI,   X,Y,Z)
C
C	Given the spherical coordinate system values of R, THETA and PHI,
C	this routine calculates the corresponding Cartesian values of
C	X,Y,Z.  THETA and PHI are assumed to be in degrees.
C
	REAL X,Y,Z,R,THETA,PHI
	REAL SIND,COSD
C
	X = R*SIND(THETA)*COSD(PHI)
	Y = R*SIND(THETA)*SIND(PHI)
	Z = R*COSD(THETA)
	RETURN
	END
C	
C
C
C========================================================================
C
      REAL*8 FUNCTION PARSETIME(TIME)
C            This function takes an integer array of 11 values and
C         calculates the approximate number of seconds since January, 1900.
C         These values are for comparison purposes only.
C	    Performance across year boundires is not optimal, esp. leap years.
C         Will fail past the year 2000.
C
      INTEGER*2 TIME(11)
C         Seconds per year
      PARSETIME = 31662400.*(10*TIME(1)+ TIME(2))
C         Seconds per day
      PARSETIME = PARSETIME + 86400.*(100*TIME(3)+10*TIME(4)+TIME(5))
C         Seconds per hour
      PARSETIME = PARSETIME + 3600.*(10*TIME(6) + TIME(7))
C         Seconds per minute
      PARSETIME = PARSETIME + 60.*(10*TIME(8) + TIME(9))
C         And finally the seconds
      PARSETIME = PARSETIME + 10.*TIME(10) +1.*TIME(11)
      RETURN
      END
C
C========================================================================
C
	CHARACTER*17 FUNCTION CTIMELTT(LTT)
C.....Converts the 11 INTEGER*2 argument LTT into the 17 character
C.....CTT time format.
	INTEGER*2 I,LTT(11)
	CTIMELTT = ' '
	WRITE(CTIMELTT,102) (LTT(I),I=1,11)
 102	FORMAT ('19',2I1,1X,3I1,1X,2I1,':',2I1,':',2I1)
	RETURN
	END
C
C========================================================================
C
	REAL*8 FUNCTION PARCTIME(CTT)
C
C	Parse a CHARACTER*17 time into a REAL*8 time.
C
	REAL*8 PARSETIME
	CHARACTER*17 CTT
	INTEGER*2 LTT(11),ZERO,I
C
	ZERO = ICHAR('0')
C.....Year.
	LTT(1)  = ICHAR(CTT(3:3))   - ZERO
	LTT(2)  = ICHAR(CTT(4:4))   - ZERO
C.....Day.
	LTT(3)  = ICHAR(CTT(6:6))   - ZERO
	LTT(4)  = ICHAR(CTT(7:7))   - ZERO
	LTT(5)  = ICHAR(CTT(8:8))   - ZERO
C.....Hour.
	LTT(6)  = ICHAR(CTT(10:10)) - ZERO
	LTT(7)  = ICHAR(CTT(11:11)) - ZERO
C.....Minute.
	LTT(8)  = ICHAR(CTT(13:13)) - ZERO
	LTT(9)  = ICHAR(CTT(14:14)) - ZERO
C.....Second
	LTT(10) = ICHAR(CTT(16:16)) - ZERO
	LTT(11) = ICHAR(CTT(17:17)) - ZERO
C
	DO 5 I = 1, 11
	   IF (LTT(I).LT.0) LTT(I) = 0
 5	CONTINUE
C
	PARCTIME = PARSETIME(LTT)
	RETURN
	END
C
	INTEGER FUNCTION IROUND(VALUE)
C
	REAL VALUE
	IF (VALUE.GE.0) THEN
	   IROUND = INT(VALUE + 0.50001)
	ELSE
	   IROUND = INT(VALUE - 0.50001)
	ENDIF
	RETURN
	END
C
C
	INTEGER FUNCTION GETINT(A,B)
C	Returns an integer read from the console in the range (A,B)
C
	INTEGER A,B,I
 10	READ(*,*,ERR=10) I
	IF ((I.LT.A).OR.(I.GT.B)) THEN
	   WRITE(*,*) ' Invalid choice, try again ...'
	   GOTO 10
	ENDIF
	GETINT = I
	WRITE(*,*) ' '
	RETURN
	END
C
C  ======================================================================
C 
	SUBROUTINE READMAPDATA(INFILE,EOF) 
C 
C===== READ THE MAP DATA FROM A FILE <infile> FOR LATER USE IN PRINTING 
C	SKYMAPS OR CALCULATING DRIFT VELOCITIES.  FREQUENCY READ IS
C	 <IFF>  THE FORMAT OF THE PREFACE IS AS FOLLOWS:
C
C	NOTE:  Range is converted to meters and Frequency to Hertz.
C 
C
C  POSITION ! FORMAT  !  INFORMATION
C_______________________________________________________________
C  1        ! A1      !  SPACE                                  !
C  2-4      ! 3Z1     !  STATION ID CODE                        !
C  5-9      ! I5      !  RECORD NO.                             !
C  10       ! A1      !  SPACE                                  !
C  11-15    ! 5Z1     !  DATE                                   !
C  16       ! A1      !  SPACE                                  !
C  17-22    ! 6Z1     !  TIME                                   !
C  23       ! 1X      !  SPACE                                  !
C  24       ! Z1      !  PROGRAM SET                            !
C  25       ! Z1      !  PROGRAM TYPE                           !
C  26-31    ! 6Z1     !  JOURNAL (6 HEX DIGITS)                 !
C  32-38    ! 7Z1     !  P1 TO P7 (7 HEX DIGITS)                !
C  39       ! 1X      !  SPACE                                  !
C  40-46    ! 7Z1     !  XLZTNRW (7 HEX DIGITS)                 !
C  47-53    ! 7Z1     !  K(I*)(G*)HEIG (7 HEX DIGITS)           !
C  54-57    ! I4      !  ZMAX (10THS OF A DEGREE)               !
C  58-59    ! I2      !  NFREQ=FREQUENCY NUMBER                 !
C  60-66    ! I7      !  FREQ(NFREQ), IN 100-HZ UNITS           !
C  67-71    ! I5      !  RANG(NFREQ), IN 100-METER UNITS        !
C  72-74    ! I3      !  IGAIN(NFREQ), IN DB                    !
C  75-77    ! I3      !  MPAMP(NFREQ)                           !
C  78-80    ! I3      !  MAXAMP(NFREQ)                          !
C  81-84    ! I4      !  NO. OF SOURCES FOR NFREQ               !
C  85-87    ! I3      !  KTH, CASE SELECTION THRESHOLD          !
C  88-90    ! I3      !  LTH, SPECTRAL LINE SELECTION THRESHOLD ! 
C================================================================
C
	INTEGER INFILE,I,I1,I2,IRNG,IFRQ,IZMAX
	CHARACTER*2 JUNK
	LOGICAL EOF
C
C	VARIABLES FOR THE COMMON BLOCKS
C
C.....For /MAPDATA/
$INCLUDE:'\DRIFT\PLAY\MAPDATA.CMN'
C
C
	DO 10 I =1, 89
 10   IPREF(I) = 0
	DO 20 I=1,256
	   XMAP(I) = 0.0
	   YMAP(I) = 0.0
	   MAPAMP(I) = 0
	   MAPRMS(I) = 0
 20   CONTINUE
	EOF = .FALSE.
C
	READ(INFILE,110,END=99) (IPREF(I),I=41,43),NMAP,
     +  (IPREF(I),I=1,19), (IPREF(I),I=26,32), (IPREF(I),I=44,57),
     +  IZMAX,IFF,IFRQ,IRNG,GAIN,MPAMP,MAXAMP,NUMSRC
C
 110	FORMAT (1X,3Z1,I5,1X,5Z1,1X,6Z1,1X,15Z1,1X,14Z1
     +	  I4,I2,I7,I5,4I3,I4) 
C
	RANGE = FLOAT(IRNG)*100.
	FREQ =  FLOAT(IFRQ)*100.
	ZMAX =  FLOAT(IZMAX)/10.
	 
	DO 140 I1=1,NUMSRC,26 
	   I2=I1+25
	   IF(I2.GT.NUMSRC) I2=NUMSRC 
	   READ(INFILE,151) JUNK,(YMAP(I),I=I1,I2) 
	   READ(INFILE,151) JUNK,(XMAP(I),I=I1,I2) 
	   READ(INFILE,150) JUNK,(MAPAMP(I),I=I1,I2)
	   READ(INFILE,150) JUNK,(MAPDOP(I),I=I1,I2) 
	   READ(INFILE,150) JUNK,(MAPRMS(I),I=I1,I2) 
  140 CONTINUE
  150 FORMAT(1X,A2,26I5)
  151 FORMAT(1X,A2,26F5.1)
C 
	RETURN
C     End of File condition.
 99   EOF = .TRUE.
 	NUMSRC = 0
	RETURN
	END
C ==========================================================================
C
	SUBROUTINE SETBOUNDS()
C
C	Sets the values for the /BOUNDS/ common block
C
C.....Parameters.
$INCLUDE:'\DRIFT\PLAY\PARAMS.CMN'
C
C.....For /BOUNDS/
$INCLUDE:'\DRIFT\PLAY\BOUNDS.CMN'
C
C	Simple boundary conditions: All ranges for frequencies from
C	0.0 MHz to 12 MHz in steps of 0.5 MHz
C
	INTEGER GETINT,IBIN
	REAL FMIN,FMAX,DF,F
C
	FMIN = 0.0
	FMAX = 12.0
	DF = 0.5
	NFRBS = 0
	WRITE(*,*) ' Frequency-Range Boundaries for source Binning.'
	WRITE(*,*) '      1) All in one bin.'
	WRITE(*,*) '      2) 0.5 MHz bins, 0.0 to 12.0 MHz, all heights.'
	IBIN = GETINT(1,2)
	IF (IBIN.EQ.1) THEN
	   NFRBS = 1
	   BFRQ(1,1) = 0.0
	   BFRQ(1,2) = 99.9
	   BRNG(1,1) = 0.0
	   BRNG(1,2) = 9999.0
	ELSE
	   DO 20 F = FMIN, FMAX, DF
	      NFRBS = NFRBS + 1
	      IF(NFRBS.GT.NFRB) THEN
	         WRITE(*,*) ' SETBOUNDS -- Not enough FRBs'
	         NFRBS = NFRB
	      ENDIF
	      NBSRC(NFRBS) = 0.0
	      BFRQ(NFRBS,1) = F
	      BFRQ(NFRBS,2) = F+DF
	      BRNG(NFRBS,1) = 0.0
	      BRNG(NFRBS,2) = 9999.0
 20	   CONTINUE
	ENDIF
	RETURN
	END
C
	CHARACTER*17 FUNCTION CTIME(ISEL)
C	
C	As a convenience feature, this function returns the CHARACTER*17 time
C	indicated by ISEL.
C
C	1) Current time (CNTIME)
C	2) Start time of current accumulation (CCSTIME)
C	3) Start time of entire run (CSTIME)
C	4) End time of entire run (CETIME)
C	5) CCSTIME, truncated to previous ATIME (In Minutes).
C	6) CCSTIME in Decimal Hours.
C	7) CCSTIME in Decimal Days.
C
$INCLUDE:'\DRIFT\PLAY\TIME.CMN'
C
	INTEGER ISEL,IAT,IROUND,ID,IH,IM,IS
	REAL RH,RM
	CHARACTER*17 CTT
C
	IF (ISEL.EQ.1) THEN
	   CTIME = CNTIME
	ELSE IF (ISEL.EQ.2) THEN
	   CTIME = CCSTIME
	ELSE IF (ISEL.EQ.3) THEN
	   CTIME = CSTIME
	ELSE IF (ISEL.EQ.4) THEN
	   CTIME = CETIME
	ELSE IF ((ISEL.EQ.5).AND.(ATIME.GT.0)) THEN
	   CTT = CCSTIME
C........# of Minutes to round off to.
	   RM = ATIME/60.
	   IAT = MAX(IROUND(RM),1)
	   READ(CTT,'(9X,I2,1X,I2,1X,I2)') IH,IM,IS
	   RM = IM + IS/60.
	   IM = IAT*INT(RM/IAT)
	   IS = 0
	   WRITE(CTT,'(A12,I2.2,A1,I2.2)') CTT,IM,':',IS
	   CTIME = CTT
	ELSE IF (ISEL.EQ.6) THEN
C........Time in YYYY DDD HH.HHHHH
	   CTT = CCSTIME
	   READ (CTT,'(9X,I2,1X,I2,1X,I2)') IH,IM,IS
	   RH = IH + IM/60. + IS/3600.
	   WRITE(CTT,'(A9,F8.5)') CTT(1:9),RH
	   CTIME = CTT
	ELSE IF (ISEL.EQ.7) THEN
C........Time in YYYY DDD.DDDDDDD
	   CTT = CCSTIME
	   READ (CTT,'(5X,I3,1X,I2,1X,I2,1X,I2)') ID,IH,IM,IS
	   RH = IH + IM/60. + IS/3600.
	   RH = ID + RH/24.0
	   WRITE(CTT,'(A4,F12.7)') CTT(1:4),RH
	   CTIME = CTT
	ELSE
	   CTIME = 'CTIME:Bad option '
	ENDIF
	RETURN
	END

